/*
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gradle.api.internal.project;

import org.gradle.api.InvalidUserDataException;
import org.gradle.api.Project;
import org.gradle.api.specs.Spec;
import org.gradle.util.HelperUtil;
import org.junit.Before;
import org.junit.Test;

import java.util.SortedSet;
import java.util.TreeSet;

import static junit.framework.Assert.assertSame;
import static org.gradle.util.WrapUtil.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.*;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * @author Hans Dockter
 */
public class DefaultProjectRegistryTest {
    public static final String CHILD_NAME = "child";
    public static final String CHILD_CHILD_NAME = "childchild";
    private DefaultProject rootMock;
    private DefaultProject childMock;
    private DefaultProject childChildMock;

    private DefaultProjectRegistry<ProjectInternal> projectRegistry;

    @Before
    public void setUp() {
        projectRegistry = new DefaultProjectRegistry<ProjectInternal>();
        rootMock = HelperUtil.createRootProject();
        childMock = HelperUtil.createChildProject(rootMock, CHILD_NAME);
        childChildMock = HelperUtil.createChildProject(childMock, CHILD_CHILD_NAME);
        projectRegistry.addProject(rootMock);
        projectRegistry.addProject(childMock);
        projectRegistry.addProject(childChildMock);
    }

    @Test
    public void addProject() {
        checkAccessMethods(rootMock, toSortedSet(rootMock, childMock, childChildMock), toSortedSet(childMock,
                childChildMock), rootMock);
        checkAccessMethods(childMock, toSortedSet(childMock, childChildMock), toSortedSet(childChildMock), childMock);
        checkAccessMethods(childChildMock, toSortedSet(childChildMock), new TreeSet(), childChildMock);
    }

    private void checkAccessMethods(Project project, SortedSet<DefaultProject> expectedAllProjects,
                                    SortedSet<DefaultProject> expectedSubProjects, Project expectedGetProject) {
        assertSame(expectedGetProject, projectRegistry.getProject(project.getPath()));
        assertEquals(expectedAllProjects, projectRegistry.getAllProjects(project.getPath()));
        assertEquals(expectedSubProjects, projectRegistry.getSubProjects(project.getPath()));
        assertSame(expectedGetProject, projectRegistry.getProject(project.getProjectDir()));
        assertTrue(projectRegistry.getAllProjects().contains(project));
    }

    @Test
    public void cannotLocateProjectsWithAmbiguousProjectDir() {
        DefaultProject duplicateProjectDirProject = HelperUtil.createChildProject(childMock, "childchild2", childMock.getProjectDir());
        projectRegistry.addProject(duplicateProjectDirProject);

        try {
            projectRegistry.getProject(childMock.getProjectDir());
            fail();
        } catch (InvalidUserDataException e) {
            assertThat(e.getMessage(), startsWith("Found multiple projects with project directory "));
        }
    }

    @Test
    public void accessMethodsForNonExistingsPaths() {
        projectRegistry = new DefaultProjectRegistry<ProjectInternal>();
        Project otherRoot = HelperUtil.createRootProject();
        assertNull(projectRegistry.getProject(otherRoot.getPath()));
        assertEquals(new TreeSet<ProjectInternal>(), projectRegistry.getAllProjects(otherRoot.getPath()));
        assertEquals(new TreeSet<ProjectInternal>(), projectRegistry.getSubProjects(otherRoot.getPath()));
        assertNull(projectRegistry.getProject(otherRoot.getProjectDir()));
    }

    @Test
    public void canLocalAllProjects() {
        assertThat(projectRegistry.getAllProjects(), equalTo(toSet((ProjectInternal) rootMock, childMock,
                childChildMock)));
    }
    
    @Test
    public void canLocateAllProjectsWhichMatchSpec() {
        Spec<Project> spec = new Spec<Project>() {
            public boolean isSatisfiedBy(Project element) {
                return element.getName().contains("child");
            }
        };

        assertThat(projectRegistry.findAll(spec), equalTo(toSet((ProjectInternal) childMock, childChildMock)));
    }

    @Test
    public void canRemoveProject() {
        String path = childChildMock.getPath();
        assertThat(projectRegistry.removeProject(path), sameInstance((ProjectInternal) childChildMock));
        assertThat(projectRegistry.getProject(path), nullValue());
        assertThat(projectRegistry.getProject(childChildMock.getProjectDir()), nullValue());
        assertTrue(projectRegistry.getAllProjects(path).isEmpty());
        assertTrue(projectRegistry.getSubProjects(path).isEmpty());
        assertFalse(projectRegistry.getAllProjects().contains(childChildMock));
        assertFalse(projectRegistry.getAllProjects(":").contains(childChildMock));
        assertFalse(projectRegistry.getSubProjects(":").contains(childChildMock));
    }
}
