<?php

/*
# Copyright 2004, Revolution Linux Inc., Nicolas Dufresne and Yves Trudeau
#
# This file is part of the MILLE-XTERM distribution.
# See the MILLE-XTERM (english) and/or the MILLE (french) project web site
#
# http://www.revolutionlinux.com/mille-xterm/
# http://www.mille.ca/
#
# The MILLE-XTERM framework is covered by the GNU General Public License. See
# the COPYING file in the top-level MILLE-XTERM directory. Software packages
# that are included in the MILLE-XTERM distribution have their own licenses.
#
# -------------------------------------------------------------------------
*/
if (is_readable('config.php')) {
	require_once 'config.php';
}
require_once 'dbFunctions.php';
require_once 'Attribute.php';
require_once 'HWGroupRule.php';
require_once 'HWGroupFinder.php';

/**
	 * Constant definition for nodetype value
	 */
	 define("NODE_TYPE",0);
	 define("COMPUTER_TYPE",1);
	 define("HWGROUP_TYPE",2);
	 define("USER_TYPE",3);
	 define("GROUP_TYPE",4);
/**
*
* This class describe a node of the tree stored in the Database
*
* @package MILLE-CONFIGURATOR
* @author Nicolas Dufresnes, Yves Trudeau, Benoit des Ligneris
*/
class Node {

	/**
	*	Nodes class constructor, starts connection.
	*/

	/**
	* Id of the node (primary key)
	* @access private
	* @var int
	*/
	var $id;
	/**
	* Id of the parent node (primary key)
	* @access private
	* @var int
	*/
	var $id_parent;

	/**
	* Leftval from nstree
	* @access private
	* @var int
	*/
	var $leftval;
	/**
	* Rightval from nstreee
	* @access private
	* @var int
	*/
	var $rightval;
	/**
	* Name of the node
	* @access private
	* @var string
	*/
	var $name;
	/**
	* MAC adress of the node (must be a real MAC adress !)
	* @access private
	* @var string
	*/
	var $mac;
	/**
	 * nodetype take the following values :
	 *    0 = node : a node in the tree, can contain childs nodes ans leafs
	 *    1 = computer : leaf with mac address
	 *    2 = hwgroup : hardware group defined with a set of rules
	 *    3 = user : user present in an external system (reference field contains username)
	 *    4 = group : group present in an external system (reference field contains groupname)
		 */
	var $nodetype;
	/**
	 * reference is the key of the entity in the external system
	 */
	var $reference;
	/**
	* False is a fatal error occured, true otherwise
	* @access private
	* @var string
	*/
	var $ok;
	/**
	* Last error (DB or logic)
	* @access private
	* @var string
	*/
	var $lastError;
	/**
	* True if node was not found on creation
	* @access private
	* @var boolean
	*/
	var $notFound;

	/**
	* The constructor of a Node
	*
	* Can be initialised in three different ways :
	* <ul>
	* <li> ID of the node
	* <li>  MAC adress of the node
	* <li> IP of the node
	* </ul>
	*
	* The first two cases read the database directly, the second one read
	* the state table and make the correspondance IP adress, MAC adress.
	* As a consequence, all IP address should be unique so that it can act
	* as another primary key for the node.
	* Example of valid initialisation :
	*
	* <code>
	*   $node=new Node(0); // Use the node ID initialisation
	*   $node=new Node(00:05:5D:FE:FC:E2); //Use the node MAC initialisation
	*   $node=new Node("192.168.2.232"); //Use the status table
	*    // (of course, IP should have been registred before !)
	* </code>
	*
	* @access public
	*/
	function Node($id = "Default") {
		global $CONFIG;
		$this->ok = true;
		$this->noFound = false;
		$defaultNodeID = false;

		if ($id == "Default") {
			if (is_numeric($CONFIG['default_node'])) {
				$id = $CONFIG['default_node'];
				$defaultNodeID = true;
			} else {
				$id = 0;
			}
		}

		if (is_numeric($id)) {
			$node = $this->getFromDB("*", "id=" . $id);
			if ($this->notFound() && $defaultNodeID) {
				$id = 0;
				$node = $this->getFromDB("*", "id=" . $id);
			}
		}
		elseif (is_array($id)) {
			$node = $id;
		}
		elseif ($this->isMac($id) && $id != "00:00:00:00:00:00") {
			$id = strtoupper($id);
			$node = $this->getFromDB("*", "mac='" . $id . "'");
		}
		elseif ($this->isIP($id)) {
			$node = $this->getFromDB("*", "S.ip='" . $id . "'", "status S INNER JOIN nodes N on S.id=N.id");
		}
		elseif ($this->isDNS($id)) {
			$id = gethostbyname($id);
			$node = $this->getFromDB("*", "S.ip='" . $id . "'", "status S INNER JOIN nodes N on S.id=N.id");
		} else {
			$node = $this->getFromDB("*", "name=" . qstr($id) . "");
			if (!$node) {
				$this->ok = false;
				$this->lastError = getMessage('bad_node_id') . " : id=" . $id;
			}
		}
		$this->id = $node['id'];
		$this->name = $node['name'];
		$this->id_parent = $node['id_parent'];
		$this->leftval = $node['leftval'];
		$this->rightval = $node['rightval'];
		$this->mac = $node['mac'];
		$this->reference = $node['reference'];
		$this->nodetype = $node['nodetype'];
	}

	///////////////////////////////////////////////////////////////////
	// Status functions
	///////////////////////////////////////////////////////////////////

	/**
	* Return true if it is node, false if it is a computer, user, group or hardware group
	*
	* @access public
	* @return boolean
	*/
	function isNode() {
		return $this->nodetype == NODE_TYPE;
	}

	/**
	* Return true if it is entity
	*
	* @access public
	* @return boolean
	*/
	function isEntity() {
		return $this->nodetype == GROUP_TYPE || $this->nodetype == USER_TYPE;
	}

	/**
	* Return true if it is computer
	*
	* @access public
	* @return boolean
	*/
	function isComputer() {
		return $this->nodetype == COMPUTER_TYPE;
	}

	/**
	* Return true if it is hwGroup
	*
	* @access public
	* @return boolean
	*/
	function isHwGroup() {
		return $this->nodetype == HWGROUP_TYPE;
	}


	/**
	* Return false if no error, true otherwise
	*
	* @access public
	* @return boolean
	*/
	function isError() {
		return !$this->ok;
	}

	/**
	* Return the last error as string
	*
	* @access public
	* @return string
	*/
	function lastError() {
		return $this->lastError;
	}

	/**
	* True if node was not found during construction, false otherwise.
	*
	* @access private
	* @return boolean
	*/
	function notFound() {
		return $this->notFound;
	}

	/**
	* True if is a MAC, false otherwise.
	*
	* @access public
	* @return boolean
	*/
	function isMac($mac) {
		return eregi("^([0-9a-f]{2}:){5}[0-9a-f]{2}$", $mac);
	}

	/**
	* True if is a IP, false otherwise.
	*
	* @access private
	* @return boolean
	*/
	function isIP($ip) {
		return ereg("^([0-2]{0,1}[0-9]{1,2}\.){3}[0-2]{0,1}[0-9]{1,2}$", $ip);
	}

	/**
	* True if is a DNS address, false otherwise.
	*
	* @access private
	* @return boolean
	*/
	function isDNS($ip) {
		return ereg("^(.*\.){2}.*$", $ip);
	}

	///////////////////////////////////////////////////////////////////
	// Database functions with error management
	///////////////////////////////////////////////////////////////////

	/**
	* Function to read node information from the DB
	*
	* @access private
	* @return mixed : NULL or assoc
	*/
	function getFromDB($column, $where, $from = "nodes", $isFatal = true) {
		if (!$result = singleResultSelect('SELECT ' . $column . ' FROM ' . $from . ' WHERE ' . $where)) {
			if (($from == "nodes" || $from == "status") && $isFatal)
				$this->lastError = getMessage('node_not_found') . " : " . $where;
			else
				$this->lastError = getMessage('not_found') . 'SELECT ' . $column . ' FROM ' . $from . ' WHERE ' . $where;
			$this->notFound = true;
			if ($isFatal)
				$this->ok = false;
			return NULL;
		}
		elseif (!is_array($result)) {
			$this->lastError = getMessage('db_error') . $result . " : " . 'SELECT ' . $column . ' FROM ' . $from . ' WHERE ' . $where;
			if ($isFatal)
				$this->ok = false;
			return NULL;
		} else
			return $result;
	}

	/**
	* Function to update node information in the DB
	*
	* @access private
	* @return boolean
	*/
	function updateDB($toSet, $table = "nodes") {
		if ($error = write("UPDATE " . $table . " SET " . $toSet . " WHERE id=" . $this->id)) {
			$this->lastError = getMessage('db_write_error') . $error . " : " . "UPDATE " . $table . " SET " . $toSet . " WHERE id=" . $this->id;
			return false;
		} else
			return true;
	}

	/**
	* Function to insert node information in the DB
	*
	* @access private
	* @return boolean
	*/
	function insertDB($data, $table = "nodes") {
		if (!$this->ok)
			return false;
		if (!is_array($data))
			return false;
		foreach ($data as $name => $value) {
			if ($list)
				$list .= ",";
			$list .= $name;
			if ($dataList)
				$dataList .= ",";
			$value = qstr($value);
			$dataList .= $value;
		}
		if ($error = write("INSERT INTO " . $table . " (" . $list . ") VALUES (" . $dataList . ")")) {
			$this->lastError = getMessage('db_write_error') . $error . " : " . "INSERT INTO " . $table . " (" . $list . ") VALUES (" . $dataList . ")";
			return false;
		} else
			return true;
	}

	/**
	* Function to update the node status in the DB
	* Please use the setStatus method for a public usage
	*
	* @access private
	* @return boolean
	*/
	function updateStatus($status, $isNew = false) {
		if (!$this->ok)
			return false;
		if (!is_array($status))
			return false;
		if ($isNew)
			return $this->insertDB($status, "status");
		foreach ($status as $name => $value) {
			if ($toSet)
				$toSet .= ",";
			$value = qstr($value);
			$toSet .= $name . "=" . $value;
		}
		return $this->updateDB($toSet, "status");
	}


	/**
	* Function to insert a log.
	* Please use the setStatus method for a public usage. Internal method
	*
	* @access private
	* @return boolean
	*/
	function insertLog($status) {
		if (!$this->ok)
			return false;
		if (!is_array($status))
			return false;
		$status['id'] = $this->id;
		$status['mac'] = $this->mac;
		unset ($status['ts_lastbootservip']);
		unset ($status['ts_lastappservip']);
		unset ($status['publickey']);
		unset ($status['termaccess']);
		$status['ts'] = UnixToDB(time());
		return $this->insertDB($status, "log");
	}
	///////////////////////////////////////////////////////////////////
	// Node getters
	///////////////////////////////////////////////////////////////////
	/**
	* get the node Id
	*
	* @access public
	* @return int
	*/
	function getID() {
		return $this->id;
	}

	/**
	* get the leftval
	*
	* @access public
	* @return int
	*/
	function getLeftval() {
		return $this->leftval;
	}

	/**
	* get the rightval
	*
	* @access public
	* @return int
	*/
	function getRightval() {
		return $this->rightval;
	}

	/**
	* get the node parent Id
	*
	* @access public
	* @return int
	*/
	function getParentID() {
		return $this->id_parent;
	}

	/**
	* get the node name
	*
	* @access public
	* @return string
	*/
	function getName() {
		return $this->name;
	}

	/**
	* get the node reference
	*
	* @access public
	* @return string
	*/
	function getReference() {
		return $this->reference;
	}

	/**
	* get the node MAC
	*
	* @access public
	* @return string
	*/
	function getMac() {
		return $this->mac;
	}

	/**
	* get the node status
	*
	* @access public
	* @return assoc
	*/
	function getStatus() {
		if (!$this->ok)
			return NULL;
		if (!$this->isComputer()) {
			$this->lastError = getMessage('cannot_get_status') . getMessage('not_computer');
			return NULL;
		}
		return $this->getFromDB("*", "id=" . $this->id, "status", false);
	}

	/**
	* get the node  childrens
	*
	* @access public
	* @return mixed : NULL or array of childrens
	*/
	function getChildrens() {
		if ($this->isError())
			return NULL;
		if (!$this->isNode())
			return NULL;
		$childrens = select('SELECT * FROM nodes WHERE id_parent=' . $this->id . ' AND nodetype IN ('.NODE_TYPE.','.COMPUTER_TYPE.','.HWGROUP_TYPE.','.GROUP_TYPE.','.USER_TYPE.') ORDER BY mac,name');

		if (!is_array($childrens) && !is_null($childrens)) {
			$this->lastError = getMessage('db_error') . $childrens;
			return NULL;
		}
		if (is_null($childrens)) {
			$this->lastError = "";
			return NULL;
		}
		foreach ($childrens as $child) {
			// Remove exception where root is root's child
			if ($child['id'] != 0)
				$childrenList[] = new Node($child);
		}
		return $childrenList;
	}

	/**
	* get the parent node
	*
	* @access public
	* @return Node
	*/
	function getParent() {
		if ($this->isError())
			return NULL;
		return new Node($this->id_parent);
	}

	/**
	* get all parents nodes
	*
	* @access public
	* @return array of Node
	*/
	function getParents() {
		$parents = select('SELECT * FROM nodes WHERE leftval<'.$this->rightval.' and rightval>'.$this->rightval.' ORDER BY leftval');
		if (!is_array($parents)) {
			return NULL;
		}
		foreach ($parents as $parent) {
			$parentList[] = new Node($parent);
		}
		return $parentList;
	}

	/**
	* Get attribute by name
	*
	* @access public
	* @return array of Attribute
	*/
	function getAttribute($name) {

		if ($this->isError())
			return NULL;
		$attributes = select("SELECT * FROM nodes N INNER JOIN attributes A ON A.nodes_id=N.id ".
								" INNER JOIN attributesDef AD ON A.attributesdef_id=AD.id " .
								" WHERE N.leftval<" . $this->rightval . " AND N.rightval>=" . $this->rightval .
								" AND AD.name=". qstr($name) ." ORDER BY N.leftval");
		if (!is_array($attributes) && !is_null($attributes)) {
			$this->lastError = getMessage('db_error') . $attributes;
			return NULL;
		}
		if (is_null($attributes)) {
			$this->lastError = "";
			return NULL;
		}
		return new Attribute($attributes[0]);
	}

	/**
	* Get all attributes implemented by this node
	*
	* @access public
	* @return array of Attribute
	*/
	function getAttributes() {
		if ($this->isError())
			return NULL;
		$attributes = select('SELECT * FROM attributes A LEFT JOIN attributesDef AD ON A.attributesdef_id=AD.id ' .
				' WHERE A.nodes_id=' . $this->id);
		if (!is_array($attributes) && !is_null($attributes)) {
			$this->lastError = getMessage('db_error') . $attributes;
			return NULL;
		}
		if (is_null($attributes)) {
			$this->lastError = "";
			return NULL;
		}
		foreach ($attributes as $attribute) {
			$attributesList[$attribute['attributesdef_id']] = new Attribute($attribute);
		}
		return $attributesList;
	}

	/**
	* Get HWGroups
	*
	* @access public
	* @return array of Attribute
	*/
	function getHwGroupsRules() {
		if ($this->isError())
			return NULL;
		if (!$this->isHwGroup()) {
			$this->lastError = getMessage('not_hwgroup');
			return NULL;
		}
		$rules = select('SELECT * FROM hwgroupsrule WHERE nodes_id=' . $this->id);
		if (!is_array($rules) && !is_null($rules)) {
			$this->lastError = getMessage('db_error') . $rules;
			return NULL;
		}
		if (is_null($rules)) {
			$this->lastError = "";
			return NULL;
		}
		foreach ($rules as $r) {
			$list[] = new HWGroupRule($r);
		}
		return $list;
	}

	/**
	* Get the computer hardware.
	*/
	function getHardware() {
		if (!$this->isComputer()) {
			$this->lastError = getMessage('cannot_set_status') . getMessage('not_computer');
			return NULL;
		}
		$hwList = select('SELECT hwkey, hwvalue FROM computershw WHERE nodes_id=' . $this->id);
		return $hwList;
	}

        /**
	*    (PDubois)Patrick Dubois - Created July 9th, 2008.
	*    
	*    This returns a boolean indicating this node belongs to a hardware group.
	*    
	*   
	*/

	function hasHWgroup(){
	   $hwgFinder = new HWGroupFinder($this, $this->getHardware());
           
           if ( ! $hwgFinder->notFound() ) {
	      return true;
	   }else
		{
		   return false;
		}
           
	}

        /**
	*    (PDubois)Patrick Dubois - Created July 9th, 2008.
	*    
	*    This returns an array of node IDs containing the IDs
	*    of all hardware groups the client may belong to.
	*
	*    
	*/

	function getHWgroupIDs(){
	   $hwgFinder = new HWGroupFinder($this, $this->getHardware());
           
           if ( ! $hwgFinder->notFound() ) {
	      return $hwgFinder->getNodesIds();
	   }else
		{
		   return $node->getID();
		}
	}
	

	/**
	* Get the herited LTS file
	*
	* This will return the complete LTS file for this node.
	*
	* @access public
	* @return assoc name => value
	*/
	function getExtendedLTS($nodesIds) {
		if (!is_array($nodesIds)) {
			return $this->getLTS();
		}
		foreach ($nodesIds as $id) {
			if (!empty($id)) {
				$inClause .= ",".$id;
			}
		}
		if (empty($inClause)) {
			return $this->getLTS();
		}
		$inClause = substr($inClause, 1);
		$attributes = select('SELECT * FROM nodes N INNER JOIN attributes A ON A.nodes_id=N.id '.
								'LEFT JOIN attributesDef AD ON A.attributesdef_id=AD.id ' .
								'WHERE (N.leftval<' . $this->rightval . ' AND N.rightval>=' . $this->rightval . ') ' .
								'OR (N.id IN (' . $inClause . ')) ' .
								'ORDER BY N.leftval');
		if (!is_array($attributes)) {
			return NULL;
		}
		# Group attributes by type
		$computerAttr=array();
		$hwgroupAttr=array();
		$groupAttr=array();
		$userAttr=array();
		foreach ($attributes as $attr) {
			$attribute = new Attribute($attr);
			switch ($attr['nodetype']) {
				case NODE_TYPE:
				case COMPUTER_TYPE:
					$computerAttr[] = $attribute;
					break;
				case HWGROUP_TYPE:
					$hwgroupAttr[] = $attribute;
					break;
				case GROUP_TYPE:
					$groupAttr[] = $attribute;
					break;
				case USER_TYPE:
					$userAttr[] = $attribute;
					break;
			}
		}
		# Build attribute list, lasts attributes overrides firsts attributes
		$allAttr = array_merge($computerAttr,$groupAttr,$userAttr,$hwgroupAttr);
		foreach ($allAttr as $attr) {
			$LTS[$attr->getName()] = $attr->getValue();
		}
		if (is_array($LTS)) {
			return $LTS;
		}
	}

	/**
	* Get the herited LTS file
	*
	* This will return the complete LTS file for this node.  The result
	* depend on parent LTS implementation
	*
	* @access public
	* @return assoc name => value
	*/
	function getLTS() {
		$attributes = select('SELECT * FROM nodes N INNER JOIN attributes A ON A.nodes_id=N.id '.
								'LEFT JOIN attributesDef AD ON A.attributesdef_id=AD.id ' .
								' WHERE N.leftval<' . $this->rightval . ' AND N.rightval>=' . $this->rightval .
								' ORDER BY N.leftval');
		if (!is_array($attributes)) {
			return NULL;
		}
		foreach ($attributes as $attr) {
			$attribute = new Attribute($attr);
			$LTS[$attribute->getName()] = $attribute->getValue();
		}
		if (is_array($LTS)) {
			return $LTS;
		}
	}

	///////////////////////////////////////////////////////////////////
	// Node setters
	///////////////////////////////////////////////////////////////////

	/**
	* Delete this node
	*
	* If is in trash delete it, otherwise, put it in trash
	*
	* @access public
	* @return bool True if success, false otherwise
	*/
	function delete() {
		startTrans();
		$rtv = $this->_delete();
		completeTrans();
		return $rtv;
	}

	/**
	* Delete this node (no transaction)
	*
	* If is in trash delete it, otherwise, put it in trash
	*
	* @access private
	* @return bool True if success, false otherwise
	*/
	function _delete() {
		if ($this->id == -1) {
			$this->lastError = getMessage('cannot_delete_trash');
			return false;
		}
		if (!$this->isNode()) {
			if ($error = delete('DELETE FROM status WHERE id=' . $this->id)) {
				$this->lastError = $error;
				return false;
			}
		}
		if ($this->isInTrash()) {
			$result = $this->_destroy();
			return $result;
		}
		$result = $this->setParent(-1);
		return $result;

	}

	/**
	* Destroy this node from DB
	*
	* @access public
	* @return True if Success
	*/
	function destroy() {
		startTrans();
		$rtv = $this->_destroy();
		completeTrans();
		return $rtv;

	}

	/**
	* Destroy this node from DB
	*
	* @access private
	* @return True if Success
	*/
	function _destroy() {
		if ($this->id == -1) {
			$this->lastError = getMessage('cannot_delete_trash');
			return false;
		}
		$ok = true;

		$childrens = $this->getChildrens();
		if (is_array($childrens)) {
			foreach ($childrens as $child) {
				if (!$ok = $child->_delete()) {
					$this->lastError = $child->lastError();
					break;
				}
			}
		}
		if ($ok) {
			$attributes = $this->getAttributes();
			if (is_array($attributes)) {
				foreach ($attributes as $attribute) {
					if (!$ok = $attribute->delete()) {
						$this->lastError = $attribute->lastError();
						break;
					}
				}
			}
		}
		if ($ok) {
			if ($error = delete('DELETE FROM nodes where id=' . $this->id)) {
				$this->lastError = $error;
				$ok = false;
			}
		}
		if ($ok) {
			$this->ok = false;
			$this->lastError = getMessage('deleted_node');
		}
		return $ok;
	}

	/**
	* Is int trash
	*
	* @access public
	* @return bool True if in trash
	*/
	function isInTrash() {
		$parents = $this->getParents();
		if (isset ($parents[1])) {
			return ($parents[1]->getID() == -1);
		}
		return false;
	}

	/**
	* set the node name
	*
	* @access public
	* @return boolean
	*/
	function setName($name) {
		if (strlen($name) == 0 && $this->isNode()) {
			$this->lastError = getMessage('empty_node_name');
			return false;
		}
		elseif (strlen($name) > 254) {
			$this->lastError = getMessage('too_long_node_name');
			return false;
		} else {
			if ($this->updateDB("name=" . qstr($name) . "")) {
				$this->name = $name;
				return true;
			}
			return false;
		}
	}

	/**
	* insert a Node. Need a name
	*
	* @access public
	* @return boolean
	*/
	function insertNode($name) {
		if ($error = write("INSERT into nodes (id_parent,name) values (" . $this->id . "," . qstr($name) . ")")) {
			$this->lastError = getMessage('db_write_error') . $error;
			return false;
		} else {
			// This won't work on another DB than PostgreSQL
			$id = singleResultSelect("SELECT currval('nodes_id_seq')");
			$id = $id['currval'];
			return $id;
		}
	}

	/**
	* insert a computer. Need a its MAC adress
	*
	* @access public
	* @return boolean
	*/
	function insertComputer($mac) {
		if ($error = write("INSERT into nodes (id_parent,mac,nodetype) values (" . $this->id . "," . qstr($mac) . "," . COMPUTER_TYPE . ")")) {
			$this->lastError = getMessage('db_write_error') . $error;
			return false;
		} else {
			return true;
		}
	}
	/**
	* insert a Node. Need a name
	*
	* @access public
	* @return boolean
	*/
	function insertHWGroup($name) {
		if ($error = write("INSERT into nodes (id_parent,name,nodetype) values (" . $this->id . "," . qstr($name) . ",".HWGROUP_TYPE.")")) {
			$this->lastError = getMessage('db_write_error') . $error;
			return false;
		} else {
			// This won't work on another DB than PostgreSQL
			$id = singleResultSelect("SELECT currval('nodes_id_seq')");
			$id = $id['currval'];
			return $id;
		}
	}
	/**
	* insert a entity (person or group)
	*
	* @access public
	* @return boolean
	*/
	function insertEntity($reference, $nodeType) {
		//The unique constraint (nodes_reference_type_unique_idx) don't do the job (may a bug in adodb)'
	    $refCounts = singleResultSelect("SELECT count(*) as refcount FROM nodes WHERE nodetype=".$nodeType."AND reference=".qstr($reference)."" );
	    $refCount = $refCounts["refcount"];
	    if($refCount == 0){
	    	// Parent is always root node id=0
			if ($error = write("INSERT into nodes (id_parent,reference,nodetype) values (" . 0 . "," . qstr($reference) . ",$nodeType)")) {
				$this->lastError = getMessage('db_write_error') . $error;
				return $this->lastError;
			} else {
				// This won't work on another DB than PostgreSQL
				$id = singleResultSelect("SELECT currval('nodes_id_seq')");
				$id = $id['currval'];
				return $id;
			}
	    }else{
	    	$this->lastError = getMessage('db_duplicate_value');
	    	return $this->lastError;
	    }
	}
	/**
	* delete a entity (person or group)
	* no trash for entity
	*
	* @access public
	* @return boolean
	*/
	function deleteEntity() {
		if($this->nodetype == GROUP_TYPE || $this->nodetype == USER_TYPE){
			$this->destroy();
		}
	}
	/**
	* move a node. Need it's parent ID
	*
	* @access public
	* @return boolean
	*/
	function setParent($id_parent) {
		if ($this->id == 0) {
			$this->lastError = getMessage('cannot_move_root');
			return false;
		}
		if ($this->id == -1) {
			$this->lastError = getMessage('cannot_delete_trash');
			return false;
		}

		$newParent = new Node($id_parent);
		if ($newParent->isError()) {
			$this->lastError = getMessage('error_moving_node') . $newParent->lastError();
			return false;
		}

		if (!$newParent->isNode()) {
			$this->lastError = getMessage('error_moving_node') . getMessage('parent_not_a_node');
			return false;
		}

		if ($this->id == $newParent->getID()) {
			$this->lastError = getMessage('error_moving_node') . getMessage('same_node_and_parent');
			return false;
		}

		$parent = $newParent->getParent();
		while ($parent->getID()) {
			if ($parent->getID() == $this->id) {
				$this->lastError = getMessage('error_moving_node') . getMessage('parent_in_node_tree');
				return false;
			}
			$parent = $parent->getParent();
		}

		if ($this->updateDB("id_parent=" . $id_parent)) {
			$this->id_parent = $id_parent;
			return true;
		} else
			return false;
	}

	/**
	* Create a child node. Need a name.
	*
	* @access public
	* @return boolean
	*/
	function createChildNode($name) {
		if ($this->isError())
			return false;
		if (!$this->isNode()) {
			$this->lastError = getMessage('error_creating_node') . getMessage('parent_not_a_node');
			return false;
		}
		if (!(strlen($name) > 0)) {
			$this->lastError = getMessage('error_creating_node') . getMessage('empty_node_name');
			return false;
		}
		if ($id = $this->insertNode($name))
			return new Node($id);
		else
			return false;
	}

	/**
	* insert a child computer. Need a its MAC adress
	*
	* @access public
	* @return boolean
	*/
	function createChildComputer($mac) {
		if ($this->isError())
			return false;
		if (!$this->isNode()) {
			$this->lastError = getMessage('error_creating_computer') . getMessage('parent_not_a_node');
			return false;
		}

		if (!$this->isMac($mac)) {
			$this->lastError = getMessage('error_creating_computer') . $mac . getMessage('not_a_mac');
			return false;
		}

		$test = new Node($mac);
		if (!$test->isError()) {
			$this->lastError = getMessage('error_creating_computer') . $mac . getMessage('computer_exists');
			return false;
		}

		if ($this->insertComputer($mac)) {
			return new Node($mac);
		} else
			return false;
	}

	/**
	* insert a child hwgroup. Need a its Name
	*
	* @access public
	* @return boolean
	*/
	function createChildHWGroup($hwgroup) {
		if ($this->isError())
			return false;
		if (!$this->isNode()) {
			$this->lastError = getMessage('error_creating_hwgroup') . getMessage('parent_not_a_node');
			return false;
		}
		if ($id = $this->insertHWGroup($hwgroup)) {
			return new Node($id);
		} else
			return false;
	}
	/**
	* Update the computer hardware.
	*/
	function setHardware($hwList) {
		if (!$this->isComputer()) {
			$this->lastError = getMessage('cannot_set_status') . getMessage('not_computer');
			return NULL;
		}
        $update = True;
        $db_hwList = $this->getHardware();

        if (count($db_hwList) == count($hwList)) {
            $update = False;
            $new_hwList = array();
            $new_db_hwList = array();

            foreach ($hwList as $hw) {
                array_push($new_hwList, $hw["hwkey"].":".$hw["hwvalue"]);
            }
            arsort($new_hwList);
            foreach ($db_hwList as $hw){
                array_push($new_db_hwList, $hw["hwkey"].":".$hw["hwvalue"]);
            }
            if (count(array_intersect($new_db_hwList,$new_hwList)) != count($new_hwList)) {
                //update db
                $update = True;
            }
        }

        if ($update) {
            $query = "DELETE FROM computershw WHERE nodes_id=" . $this->id . ";";

            //delete("DELETE FROM computershw WHERE nodes_id=" . $this->id);
            foreach ($hwList as $hw) {
                $query .= "INSERT into computershw (nodes_id,hwkey,hwvalue) values (" . $this->id . "," . qstr($hw["hwkey"]) . "," . qstr($hw["hwvalue"]) . ");";
                //write("INSERT into computershw (nodes_id,hwkey,hwvalue) values (" . $this->id . "," . qstr($hw["hwkey"]) . "," . qstr($hw["hwvalue"]) . ")");
            }
            //Here I use delete so the statement is executed all in once. this add the values as well.
            delete($query);
        }
	}

	/**
	* Update the status of a computer.
	*
	* Need some information.
	* <ul>
	* <LI> code 1 : Computer has booted. Need bootserverip, ip
	* <LI> code 2 : Computer has an app server. Need appserverip, plus previous
	* <LI> code 3 : User login. Need display, username, plus previous
	* <LI> code 4 : User logout. Need previous
	* </UL>
	* <code>
	* Example code
	*   $node = new Node($mac);
	*   $status = $node->getStatus();
	*   $status['code'] = 1;
	*   $status['bootservip'] = "192.168.2.2";
	*   $status['ip'] = "192.168.100";
	*   if ($node->setStatus($status)) {
	*	print "Good job!!\n";
	*   }
	*   else
	*   {
	*	print "Oops, try again...\n";
	*   }
	* </code>
	*
	* @access public
	* @return boolean
	*/
	function setStatus($status, $log = true) {
		if (!$this->ok)
			return false;
		if (!$this->isComputer()) {
			$this->lastError = getMessage('cannot_set_status') . getMessage('not_computer');
			return NULL;
		}
		// DB error will be trapped when writing
		$realStatus = $this->getStatus();

		// Validating
		if (!$this->validateStatus($status))
			return false;

		// Defaulting config
		if (!is_numeric($_CONFIG['bootServRetry_time']))
			$_CONFIG['bootServRetry_time'] = 45;
		if (!is_numeric($_CONFIG['appServRetry_time']))
			$_CONFIG['appServRetry_time'] = 10;

		if (!$realStatus) {
			$noRealStatus = true;
			$status['id'] = $this->id;
			$status['mac'] = $this->mac;
			$realStatus['ts_lastbootservip'] = 0;
			$realStatus['ts_lastappservip'] = 0;
			$realStatus['bootservretry'] = 0;
			$realStatus['appservretry'] = 0;
		} else {
			$noRealStatus = false;
			unset ($status['id']);
			unset ($status['mac']);
		}
		switch ($status['code']) {
			case 1 :
				if ((time() - $realStatus['ts_lastbootservip']) < $_CONFIG['bootServRetry_time']) {
					$status['bootservretry'] = $realStatus['bootservretry'] + 1;
				} else {
					$status['bootservretry'] = 1;
				}
				// Initialising data
				$status['appservip'] = "0.0.0.0";
				$status['display'] = '-1';
				$status['username'] = "";
				$status['appservretry'] = 0;
				$status['ts_lastappservip'] = UnixToDB(0);
				if (strtotime($status['ts_lastbootservip'])+10 >= time()) $skip=1;
				$status['ts_lastbootservip'] = UnixToDB(time());
				$status['termaccess'] = 0;
				
				// Remove upsalete status
				if ($noRealStatus) {
					delete("DELETE FROM status WHERE ip=" . qstr($status['ip']) . "");
				} else {
					delete("DELETE FROM status WHERE ip=" . qstr($status['ip']) . " and id <> " . $this->id . "");
				}

				break;
			case 2 :
				if ((time() - $realStatus['ts_lastappservip']) < $_CONFIG['appServRetry_time']) {
					$status['appservretry'] = $realStatus['appservretry'] + 1;
				} else {
					$status['appservretry'] = 1;
				}
				// Initialising data
				$status['username'] = "";
				$status['display'] = '-1';
				$status['ts_lastappservip'] = UnixToDB(time());
				$status['termaccess'] = 0;
				break;
			case 3 :
                                write("UPDATE status SET username=Null WHERE username='".$status['username']."' AND ip!='".$status['ip']."'");
				break;
			case 4 :
				$status['termaccess'] = 0;
				break;
		}

		if ($skip == 1)
			return true;
		if (!$this->updateStatus($status, $noRealStatus))
			return false;
		if ($log)
			return $this->insertLog($status);
		return true;
	}

	/**
	* Validate the status receive
	*
	* This receive a status and validate that all data
	* are valide.
	*
	* @access private
	* @return bool True if valide
	*/
	function validateStatus(& $status) {
		switch ($status['code']) {
			case 4 :
			case 3 :
				if (strlen($status['username']) == 0) {
					$this->lastError = getMessage('cannot_set_status') . getMessage('empty_username');
					return false;
				}
				if ($status['display'] == '-1') {
					$this->lastError = getMessage('cannot_set_status') . getMessage('bad_status') . "display=" . $status['display'];
					return false;
				}
			case 2 :
				if (!$this->isIP($status['appservip']) || $status['appservip'] == "0.0.0.0") {
					$this->lastError = getMessage('cannot_set_status') . getMessage('bad_ip') . "appServIp=" . $status['appservip'];
					return false;
				}
			case 1 :
				if (!$this->isIP($status['ip'])) {
					$this->lastError = getMessage('cannot_set_status') . getMessage('bad_ip') . "ip=" . $status['ip'];
					return false;
				}
				if (!$this->isIP($status['bootservip'])) {
					$this->lastError = getMessage('cannot_set_status') . getMessage('bad_ip') . "bootservip=" . $status['bootservip'];
					return false;
				}
				break;
			default :
				$this->lastError = getMessage('cannot_set_status') . getMessage('bad_number') . "code=" . $status['code'];
				return false;
				break;
		}
		return true;
	}

} // class Nodes
?>
