# ifndef _SKIT_MACROS_H
# define _SKIT_MACROS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
//    macros
//
//    author: P. Saramito
//
//    date: 16 january 1997
//
namespace rheolef { 
typedef std::vector<int>::size_type Index;

inline Index min_value(const Index&) { return 0; }
inline Index max_value(const Index&) { return std::numeric_limits<Index>::max(); }

inline int   min_value(const int&)   { return std::numeric_limits<int>::min(); }
inline int   max_value(const int&)   { return std::numeric_limits<int>::max(); }

inline float  min_value(const float&)  { return std::numeric_limits<float>::min(); }
inline float  max_value(const float&)  { return std::numeric_limits<float>::max(); }

inline Float  min_value(const Float&)  { return std::numeric_limits<Float>::min(); }
inline Float  max_value(const Float&)  { return std::numeric_limits<Float>::max(); }

// ==================================[ numerics ]=====================================

// avoid problem with abs and fabs...
template <class T>
inline T xabs (T x) { return (x >= T(0) ? x : -x); }
template<>
inline Index xabs (Index x)  { return x; }

#ifndef _RHEOLEF_HAVE_ISINF_DOUBLE
#   if defined(_RHEOLEF_HAVE_FINITE_DOUBLE) && defined(_RHEOLEF_HAVE_ISNAN_DOUBLE)
      template <class T>
      inline bool xisinf (T x) { return ! (finite (x) || isnan (x)); }
#   else
      template <class T>
      inline bool xisinf (T x) { return x >= max_value(x) || x <= -max_value(x); }
#   endif
#else // ! _RHEOLEF_HAVE_ISINF_DOUBLE
    template <class T>
    inline bool xisinf (T x) { return isinf(x); } 
#endif // _RHEOLEF_HAVE_ISINF_DOUBLE

#ifdef _RHEOLEF_HAVE_DOUBLEDOUBLE
    template <>
    inline bool xisinf (doubledouble x) { return x != x; } 
#endif // ! _RHEOLEF_HAVE_DOUBLEDOUBLE

template<>
inline bool xisinf (int x) { return false; } 
template<>
inline bool xisinf (Index x) { return false; } 


#ifndef _RHEOLEF_HAVE_ISNAN_DOUBLE
template <class T>
inline bool xisnan (T x) { return false; } 
# else
template <class T>
inline bool xisnan (T x) { return isnan(x); }
#endif
template<>
inline bool xisnan (int x) { return false; } 
template<>
inline bool xisnan (Index x) { return false; } 

template <class T>
inline bool xfinite (T x) { return !xisinf(xabs(x)) && !xisnan(xabs(x)); }

template <class T>
inline bool xisint (T x) { return Float(int(x)) == x; } 
template<>
inline bool xisint (int x) { return true; } 
template<>
inline bool xisint (Index x) { return true; } 

template<class T>
inline T  xmax (T x, T y)  { return (x > y ? x : y); }
template<class T>
inline T  xmin (T x, T y)  { return (x > y ? y : x); }

// TODO: loops may be put in a macro.c file ! (if not code will bulrp)

#ifdef TO_CLEAN
//use copy(a,n,lambda) !
template <class InputIterator, class OutputIterator, class Size>
inline
void
Copy (InputIterator a, OutputIterator b, Size n) { 
	if (!n) return; 
	InputIterator  *p = a+n-1; 
	OutputIterator *q = b+n-1; 
	while (p >= a) *q-- = *p--; 
}

//use fill_n(a,n,lambda) !
template <class T1, class T2>
inline void Set (T1* a, Index n, T2 lambda)
	{ if (!n) return; T1 *p = a+n-1; while (p >= a) *p-- = lambda; }
#endif // TO_CLEAN

template <class T>
inline
T* 
Dup (const T* a, Index n)
	{ if (!n) return 0; const T *b = new_tab_macro(T,n); copy(a,a+n,b); return b; }

template <class InputIterator1, class InputIterator2, class OutputIterator, class Size>
inline
void
Cat (InputIterator1 a, Size n1, InputIterator2 b, Size n2, OutputIterator c)
	{ copy(a,a+n1,c); copy(b,b+n2,c+n1); }

}// namespace rheolef
#endif // _SKIT_MACROS_H

