// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framepseudocolor.h"
#include "fitsimage.h"
#include "ps.h"

#include "sigbus.h"

FramePseudoColor::FramePseudoColor(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : FrameBase(i,c,item), Frame(i,c,item), FramePseudo(i,c,item)
{
}

void FramePseudoColor::buildXImage(XImage* xmap, Coord::InternalSystem sys)
{
  // we need a colorScale before we can render
  if (!validColorScale())
    return;

  // make sure we have an image
  if (!context->cfits)
    return;

  int& width = xmap->width;
  int& height = xmap->height;
  int& bytesPerLine = xmap->bytes_per_line;
  char* data = XImageData(xmap);
  unsigned long bgc = getColor(bgColorName);
  unsigned long nanc = getColor(nanColorName);

  // set bg
  memset(data, bgc, bytesPerLine * height);

  // basics
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->colors();

  FitsImage* sptr = context->cfits;
  int mosaic = isMosaic();

  // variable
  double* mm = sptr->matrixToData(sys).mm();
  FitsBound* params = sptr->getDataParams(context->frScale.scanMode());
  int srcw = sptr->width();

  double ll = sptr->getLowDouble();
  double hh = sptr->getHighDouble();
  double diff = hh - ll;

  // main loop
  SETSIGBUS
  for (long jj=0; jj<height; jj++) {
    char* dest = data + jj*bytesPerLine; // may be padded at the end

    for (long ii=0; ii<width; ii++,dest++) {

      if (mosaic) {
	sptr = context->cfits;

	mm = sptr->matrixToData(sys).mm();
	params = sptr->getDataParams(context->frScale.scanMode());
	srcw = sptr->width();

	ll = sptr->getLowDouble();
	hh = sptr->getHighDouble();
	diff = hh - ll;
      }

      do {
	double xx = ii*mm[0] + jj*mm[3] + mm[6];
	double yy = ii*mm[1] + jj*mm[4] + mm[7];

	if (xx>=params->xmin && xx<params->xmax && 
	    yy>=params->ymin && yy<params->ymax) {
	  double value = sptr->getValueDouble(long(yy)*srcw + long(xx));
       
	  if (isfinite(value)) {
	    if (value <= ll)
	      *dest = table[0];
	    else if (value >= hh)
	      *dest = table[length];
	    else
	      *dest = table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    *dest = nanc;

	  break;
	}
	else {
	  if (mosaic) {
	    sptr = sptr->nextMosaic();

	    if (sptr) {
	      mm = sptr->matrixToData(sys).mm();
	      params = sptr->getDataParams(context->frScale.scanMode());
	      srcw = sptr->width();

	      ll = sptr->getLowDouble();
	      hh = sptr->getHighDouble();
	      diff = hh - ll;
	    }
	  }
	}
      }
      while (mosaic && sptr);
    }
  }
  CLEARSIGBUS
}

// Commands

void FramePseudoColor::colormapCmd(int id, float b, float c, int i, 
				    unsigned short* index, 
				    unsigned char* cells, int cnt)
{
  cmapID = id;
  bias = b;
  contrast = c;
  invert = i;

  updateColorCells(index, cells, cnt);
  updateColorScale();
}

void FramePseudoColor::colormapMotionCmd(int id, float b, float c, int i, 
					 unsigned short* index, 
					 unsigned char* cells, int cnt)
{
  // just remember for getColorbarCmd()
  cmapID = id;
  bias = b;
  contrast = c;
  invert = i;
}

void FramePseudoColor::colormapEndCmd()
{
  update(BASE); // always update
}
