/******************************************************************************
*       SOFA, Simulation Open-Framework Architecture, version 1.0 beta 4      *
*                (c) 2006-2009 MGH, INRIA, USTL, UJF, CNRS                    *
*                                                                             *
* This library is free software; you can redistribute it and/or modify it     *
* under the terms of the GNU Lesser General Public License as published by    *
* the Free Software Foundation; either version 2.1 of the License, or (at     *
* your option) any later version.                                             *
*                                                                             *
* This library is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or       *
* FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License *
* for more details.                                                           *
*                                                                             *
* You should have received a copy of the GNU Lesser General Public License    *
* along with this library; if not, write to the Free Software Foundation,     *
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.          *
*******************************************************************************
*                               SOFA :: Modules                               *
*                                                                             *
* Authors: The SOFA Team and external contributors (see Authors.txt)          *
*                                                                             *
* Contact information: contact@sofa-framework.org                             *
******************************************************************************/
#ifndef SOFA_COMPONENT_MISC_EVALSURFACEDISTANCE_INL
#define SOFA_COMPONENT_MISC_EVALSURFACEDISTANCE_INL

#include "EvalSurfaceDistance.h"
#include <sofa/defaulttype/DataTypeInfo.h>
#include <sofa/simulation/common/Node.h>
#include <sofa/helper/gl/template.h>

#include <fstream>

namespace sofa
{

namespace component
{

namespace misc
{

template<class DataTypes>
EvalSurfaceDistance<DataTypes>::EvalSurfaceDistance()
: maxDist( initData(&maxDist, (SReal)1.0, "maxDist", "alarm distance for proximity detection"))
, pointsCM(NULL)
, surfaceCM(NULL)
, intersection(NULL)
, detection(NULL)
{
}

template<class DataTypes>
EvalSurfaceDistance<DataTypes>::~EvalSurfaceDistance()
{
    if (intersection)
        delete intersection;
    if (detection)
        delete detection;
}

//-------------------------------- init ------------------------------------
template<class DataTypes>
void EvalSurfaceDistance<DataTypes>::init()
{
    Inherit::init();
    if (!this->mstate1 || !this->mstate2)
        return;
    sofa::core::objectmodel::BaseContext* c1 = this->mstate1->getContext();
    c1->get(pointsCM);
    if (pointsCM == NULL)
    {
	serr << "EvalSurfaceDistance ERROR: object1 PointModel not found."<<sendl;
	return;
    }
    sofa::core::objectmodel::BaseContext* c2 = this->mstate2->getContext();
    c2->get(surfaceCM);
    if (surfaceCM == NULL)
    {
	serr << "EvalSurfaceDistance ERROR: object2 TriangleModel not found."<<sendl;
	return;
    }

    intersection = new sofa::component::collision::NewProximityIntersection;
    intersection->setContext(getContext());
    intersection->init();

    detection = new sofa::component::collision::BruteForceDetection;
    detection->setContext(getContext());
    detection->init();
}

//-------------------------------- eval------------------------------------
template<class DataTypes>
SReal EvalSurfaceDistance<DataTypes>::eval()
{
    if (!this->mstate1 || !this->mstate2 || !surfaceCM || !pointsCM || !intersection || !detection) return 0.0;

    const VecCoord& x0 = *this->mstate1->getX0();
    const VecCoord& x1 = *this->mstate1->getX();
    surfaceCM->computeBoundingTree(6);
    pointsCM->computeBoundingTree(6);
    intersection->setAlarmDistance(maxDist.getValue());
    intersection->setContactDistance(0.0);
    detection->setInstance(this);
    detection->setIntersectionMethod(intersection);
    sofa::helper::vector<std::pair<sofa::core::CollisionModel*, sofa::core::CollisionModel*> > vectCMPair;
    vectCMPair.push_back(std::make_pair(surfaceCM->getFirst(), pointsCM->getFirst()));

    detection->beginNarrowPhase();
    sout << "narrow phase detection between " <<surfaceCM->getClassName()<< " and " << pointsCM->getClassName() << sendl;
    detection->addCollisionPairs(vectCMPair);
    detection->endNarrowPhase();

    /// gets the pairs Triangle-Line detected in a radius lower than maxDist
    const core::componentmodel::collision::NarrowPhaseDetection::DetectionOutputMap& detectionOutputs = detection->getDetectionOutputs();

    core::componentmodel::collision::NarrowPhaseDetection::DetectionOutputMap::const_iterator it = detectionOutputs.begin();
    core::componentmodel::collision::NarrowPhaseDetection::DetectionOutputMap::const_iterator itend = detectionOutputs.end();

    xproj = x1;
    sofa::helper::vector<Real> dmin(xproj.size());
    std::fill(dmin.begin(),dmin.end(),(Real)(2*maxDist.getValue()));

    while (it != itend)
    {
	const ContactVector* contacts = dynamic_cast<const ContactVector*>(it->second);
	if (contacts != NULL)
	{
	    sout << contacts->size() << " contacts detected." << sendl;
	    for (unsigned int i=0; i<contacts->size(); i++)
	    {
		if ((*contacts)[i].elem.first.getCollisionModel() == surfaceCM)
		{
		    if ((*contacts)[i].elem.second.getCollisionModel() == pointsCM)
		    {
			int pi = (*contacts)[i].elem.second.getIndex();
			if ((*contacts)[i].value < dmin[pi])
			{
			    dmin[pi] = (Real)((*contacts)[i].value);
			    xproj[pi] = (*contacts)[i].point[0];
			}
		    }
		}
		else
		    if ((*contacts)[i].elem.second.getCollisionModel() == surfaceCM)
		{
		    if ((*contacts)[i].elem.first.getCollisionModel() == pointsCM)
		    {
			int pi = (*contacts)[i].elem.first.getIndex();
			if ((*contacts)[i].value < dmin[pi])
			{
			    dmin[pi] = (Real)((*contacts)[i].value);
			    xproj[pi] = (*contacts)[i].point[1];
			}
		    }
		}
	    }
	}
	it++;
    }
    return this->doEval(x1, xproj, x0);
}

//-------------------------------- draw------------------------------------
template<class DataTypes>
void EvalSurfaceDistance<DataTypes>::draw()
{
    if (!this->f_draw.getValue())
        return;
    if (!this->mstate1 || !this->mstate2 || xproj.empty()) return;
    const VecCoord& x1 = *this->mstate1->getX();
    const VecCoord& x2 = xproj; //*this->mstate2->getX();
    this->doDraw(x1, x2);
}

} // namespace misc

} // namespace component

} // namespace sofa

#endif
