#!/usr/bin/env python

__authors__ = "Martin Sandve Alnes"
__date__ = "2008-10-15 -- 2008-10-15"

import unittest

import ufl
from ufl import *
from ufl.algorithms import expand_compounds

import sfc as sfc
from sfc.representation.swiginac_eval import SwiginacEvaluator

from sfc.symbolic_utils import symbolic_matrix, symbolic_vector 

class MockCell:
    def __init__(self, nsd):
        self.nsd = nsd

class MockFormRep:
    def __init__(self, nsd):
        self.rank = 0
        self.nsd = nsd
        self.cell = MockCell(nsd)

class MockVariables:
    def __init__(self, nsd):
        self.x = symbolic_vector(nsd, "x")
        self.Ginv_sym = symbolic_matrix(nsd, nsd, "Ginv")

def ufl2swiginac(expression, nsd):
    formrep = MockFormRep(nsd)
    variables = MockVariables(nsd)
    use_symbols = False
    s = SwiginacEvaluator(formrep, variables, use_symbols)
    return s.visit(expression)

class Ufl2SwiginacTest(unittest.TestCase):
    def __init__(self, *args, **kwargs):
        unittest.TestCase.__init__(self, *args, **kwargs)
    
    def setUp(self):
        pass
    
    def tearDown(self):
        pass
    
    def testSetup(self):
        pass
    
    def testIndexed1(self):
        A = as_vector((1,2,3))
        a = A[0] + A[1] +A[2]
        b = ufl2swiginac(a, 3)
        c = 1 + 2 + 3
        self.assertEqual(c, b)
    
    def testIndexed2(self):
        A = as_matrix(((1,2),(3,4)))
        a = sum(A[i,j] for i in (0,1) for j in (0,1))
        b = ufl2swiginac(a, 2)
        c = 1 + 2 + 3 + 4
        self.assertEqual(c, b)
    
    def testIndexed3(self):
        A = as_tensor(((1,2,3), (4,5,6), (7,8,9)))
        a = A[i,i]
        b = ufl2swiginac(a, 3)
        c = 1+5+9
        self.assertEqual(c, b)
    
    def testProduct1(self):
        A = as_vector((1,2,3))
        a = A[0]*A[1]*A[2]
        b = ufl2swiginac(a, 3)
        c = 1 * 2 * 3
        self.assertEqual(c, b)
    
    def testProduct2(self):
        u = as_vector((2,3,5))
        v = as_vector((7,11,13))
        a = u[i]*v[i]
        b = ufl2swiginac(a, 3)
        c = 2*7 + 3*11 + 5*13
        self.assertEqual(c, b)
    
    def testProduct3(self):
        u = as_tensor(((1,2,3), (4,5,6), (7,8,9)))
        v = as_tensor(((4,5,6), (1,2,3), (4,5,6)))
        a = u[i,j]*v[i,j]
        b = ufl2swiginac(a, 3)
        c = (4+10+18)*2 + 7*4+8*5+9*6
        self.assertEqual(c, b)

    def testProduct4(self):
        u = as_tensor(((1,2,3), (4,5,6), (7,8,9)))
        v = as_tensor(u[i,j], (j,i))
        a = inner(u, v.T)
        a = expand_compounds(a, 3)
        b = ufl2swiginac(a, 3)
        c = sum(ii**2 for ii in range(10)) 
        self.assertEqual(c, b)
    
    def testSpatialDiff1(self):
        "Test a single constant dx"
        A = as_vector((2,3))
        a = A[0].dx(0)
        b = ufl2swiginac(a, 2)
        c = 0
        self.assertEqual(c, b)
    
    def testSpatialDiff2(self):
        "Test a single div-type df_i/dx_i"
        A = as_vector((2,3))
        a = A[i].dx(i)
        b = ufl2swiginac(a, 2)
        c = 0
        self.assertEqual(c, b)
    
    def testSpatialDiff3(self):
        "Test a double div-type df_ij/dx_ij"
        A = as_vector(((2,3),(4,5)))
        a = A[i,j].dx(j,i)
        b = ufl2swiginac(a, 2)
        c = 0
        self.assertEqual(c, b)
        

tests = [Ufl2SwiginacTest]

if __name__ == "__main__":
    unittest.main()
