/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

///////////////////////////////////////////////////////////////////////////////
//
// Class:			UserNotificationThread
// Description:		This class is the thread that does the actual User Notification
//                  work, i.e. it subscribing AMT for recieving events and waiting
//                  for events to come.
//
///////////////////////////////////////////////////////////////////////////////
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "UserNotificationThread.h"
#include "Credentials.h"
#include "AMTPPSharedData.h"
#include "AMTPPDefs.h"
#include "UNSEventLogger.h"
#include "PTHICommand.h"

// Set the static constants values.
const char *	UserNotificationThread::IP_LOCALHOST = "127.0.0.1";
const unsigned long UserNotificationThread::DEFAULT_CLIENT_INTERVAL = 10 * SECOND;
const unsigned long UserNotificationThread::DEFAULT_BIND_INTERVAL	= 5 * SECOND;
const unsigned long UserNotificationThread::HECI_DOWNTIME_ON_RESUME = 10 * SECOND;
const unsigned long UserNotificationThread::UNS_PROV_WAIT_TIME = 1 * MINUTE;

//*****************************************************************************
// Name			: UserNotificationThread
// Description	: Constructor
//*****************************************************************************
UserNotificationThread::UserNotificationThread(UNSEventLogger *pLogger/*=NULL*/) :
	PeriodicActionThread("UserNotification", DEFAULT_CLIENT_INTERVAL, HECI_DOWNTIME_ON_RESUME)
{
	m_clientIP = IP_LOCALHOST; //default client ip is local host - 127.0.0.1
	m_serverIP = IP_LOCALHOST; //default server ip is local host - 127.0.0.1

	// Set the logger for client and server.
	m_pLogger = pLogger;
	m_client.SetLogger(pLogger);
	m_server.SetLogger(pLogger);

	// The first state is SERVER_BIND_STATE, no delay is needed here.
	m_State = SERVER_BIND_STATE;
	setActionInterval(0);

	// Generate credentials.
	m_client.GenerateCredentials(); //  Generate credentials for soap server.
	m_server.SetCredentials(m_client.GetUserid(), m_client.GetPasswd());
}

//*****************************************************************************
// Name			: UserNotificationThread
// Description	: Destructor
//*****************************************************************************
UserNotificationThread::~UserNotificationThread()
{

}

void UserNotificationThread::SetLogger(UNSEventLogger *pLogger)
{
	m_pLogger = pLogger;
	m_client.SetLogger(pLogger);
	m_server.SetLogger(pLogger);
}

//*****************************************************************************
// Name			: UserNotificationThread::init
// Description	:
//*****************************************************************************
void UserNotificationThread::init()
{
	setActionInterval(0);
	m_client.SetIP(m_clientIP);
}

//*****************************************************************************
// Name			: UserNotificationThread::SetClientIP
// Description	:
//*****************************************************************************
void UserNotificationThread::SetClientIP(const string &clientIP)
{
	m_clientIP = clientIP;
}

//*****************************************************************************
// Name			: UserNotificationThread::SetClientIP
// Description	:
//*****************************************************************************
void UserNotificationThread::SetServerIP(const string &serverIP)
{
	m_serverIP = serverIP;
}

//*****************************************************************************
// Name			: UserNotificationThread::SetCredentials
// Description	: Set username and password for connecting AMT.
//*****************************************************************************
void UserNotificationThread::SetCredentials(const Credentials &cred)
{
	m_client.SetCredentials(cred);
}

//*****************************************************************************
// Name			: UserNotificationThread::Cleanup
// Description	: Remove this client subscription from AMT.
//*****************************************************************************
void UserNotificationThread::cleanup()
{
	// Cencel client subscription.
	if (m_client.CancelAlertSubscription(NULL) != UNS_STATUS_SUCCESS)
	{
		if (m_pLogger)
			m_pLogger->CancelSubscriptionError();
	}
	m_pLogger = NULL;
	printf ("Cleanup: Client subscription removed\n");
}

//*****************************************************************************
// Name			: UserNotificationThread::BindServer
// Description	: Bind the server to the first available port.
//*****************************************************************************
UserNotificationThread::State UserNotificationThread::BindServer()
{
	// First bind the server
	if (!m_server.IsInit())
	{
		if (m_server.Bind(m_serverIP.c_str()) != UNS_STATUS_SUCCESS)
		{
			if (m_pLogger)
				m_pLogger->BindError();
			printf ("Failed to bind to %s...\n", m_serverIP.c_str());
			setActionInterval(DEFAULT_BIND_INTERVAL);
			return SERVER_BIND_STATE;
		}
	}
	return SUBSCRIBE_FOR_ALERT_STATE;
}

//*****************************************************************************
// Name			: UserNotificationThread::SubscribeForAlert
// Description	: Subscribe for alert at the local AMT machine.
//*****************************************************************************
UserNotificationThread::State UserNotificationThread::SubscribeForAlert(bool bLogFailure)
{
	PTHICommand command(false);
	AMT_PROVISIONING_STATE provState;
	if (command.PTHIClient.Init() &&
		(command.GetProvisioningState(&provState) == PT_STATUS_SUCCESS) &&
		(provState == PROVISIONING_STATE_POST))
	{	
	// Subscribe for alert.
	if (!m_client.IsSubscribed()) {
		if (m_client.SubscribeForGeneralAlert(m_server.GetPort()) != UNS_STATUS_SUCCESS)
		{
			if (bLogFailure && m_pLogger)
				m_pLogger->SubscriptionError();
			printf("Failed to subscribe AMT local machine...\n");
			setActionInterval(DEFAULT_CLIENT_INTERVAL);
			return SUBSCRIBE_FOR_ALERT_FAILURE_STATE;
		}
	}
	if (m_pLogger)
		m_pLogger->SubscriptionSuccess();
	setActionInterval(0);
	return SERVER_RUN_STATE;
	}
	else
	{
		setActionInterval(UNS_PROV_WAIT_TIME);
		return SUBSCRIBE_FOR_ALERT_STATE;
	}
}

//*****************************************************************************
// Name			: UserNotificationThread::SubscribeForAlertFailure
// Description	: Subscribe for alert at the local AMT machine (no message).
//*****************************************************************************
UserNotificationThread::State UserNotificationThread::SubscribeForAlertFailure()
{
	return SubscribeForAlert(false);
}

//*****************************************************************************
// Name			: UserNotificationThread::RunServer
// Description	: Run the server.
//*****************************************************************************
UserNotificationThread::State UserNotificationThread::RunServer()
{
	m_server.RunServer();
	return SERVER_RUN_STATE;
}

//*****************************************************************************
// Name			: UserNotificationThread::action
// Description	: Perform a single thread action according to its state.
//*****************************************************************************
void UserNotificationThread::action()
{
	switch (m_State)
	{
		case SERVER_BIND_STATE:
			m_State = BindServer();
			break;
		case SUBSCRIBE_FOR_ALERT_STATE:
			m_State = SubscribeForAlert();
			break;
		case SUBSCRIBE_FOR_ALERT_FAILURE_STATE:
			m_State = SubscribeForAlertFailure();
			break;
		case SERVER_RUN_STATE:
			m_State = RunServer();
			break;
	}
}
