# -*- coding: utf-8 -*-
#############################################
#  Rafael Proença     <cypherbios@ubuntu.com>
#  Laudeci Oliveira    <laudeci@gmail.com>
#
#  Copyright 2006 APTonCD DevTeam.
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published
#  by the Free Software Foundation; version 2 only.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#############################################
import gobject
import gtk
import pygtk
import pango

from APTonCD.core import constants
from APTonCD.core.utils import get_icon
from APTonCD.core.gui import processEvents

# Update the value of the progress bar so that we get
# some movement
def progress_timeout(pbobj):
    pbobj.pulse()
    processEvents()
    # As this is a timeout function, return TRUE so that it
    # continues to get called
    return True

class ProgressDialogBase(gtk.Window):
    """
        Returns a window that contains a number of properties to
       access what a common Progress window should have accordling to HIG.
    """
    def __init__(self, parent = None, title = constants.MESSAGE_0052, description = constants.MESSAGE_0053, task=constants.MESSAGE_0054):
        """
            Creates a progress window with some given properties. 
            
            @title is the text showed in the Window Title and progress Title.
            @description is the text showed under progress title generally informing what the process is 
               about to do.
            @parent is the widget who will own this window.
            @the task name showed under the progress bar.
            
        """
        # Creating the windows and its properties
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        
        
        self.parentui = parent
        
        if self.parentui != None:
           # set whon this window will be answer to, and disabel the father window
           if isinstance(parent, gtk.Window):
               self.set_transient_for(parent)
               self.parentui.set_sensitive(False)
           else:    
               gtk.gdk.Window.set_transient_for(self.get_root_window(), parent) 
        self.set_position(gtk.WIN_POS_CENTER_ALWAYS)
        
        
        self._create_ui(title, description, task)
        self._connect_events()
        
        self.timer = 0 
        
        #it will hold cancel status
        self.cancel_status = False

    def _connect_events(self):
        """ 
            prevent the window from closing with the delete button 
            (there is a cancel button in the window)
        """
        self.connect("delete_event", lambda e, w: True);
        self.connect('destroy', self.on_destroy)
    
    def on_destroy(self, *args):
        if self.parentui != None:
            if isinstance(self.parentui, gtk.Window):
                self.parentui.set_sensitive(False)
    
    def _create_ui(self, title, description, task):    
        """
            Create and setup widgets properties.
            
            @title is the text showed in the Window Title and progress Title.
            @description is the text showed under progress title generally informing what the process is 
               about to do.
            @the task name showed under the progress bar.
        """
        # window properties
        self.set_border_width(6)
        self.set_resizable(False)
        self.set_skip_taskbar_hint(True) #conforming to HIG, progress window must show in taskbar
        #self.set_size_request(520, 190)
        self.set_title(title)
        self.set_icon_name(constants.APP_ICON_NAME)
        self.realize()
        self.window.set_functions(gtk.gdk.FUNC_MOVE)
        
        #main box where widgets will be in
        _vbox = gtk.VBox()
        _vbox.set_border_width(6)
        self.add(_vbox)
        
        _hbox = gtk.HBox(homogeneous = False, spacing = 12)
        _vbox.pack_start(_hbox, False)
        
        _image = gtk.Image()
        _image.set_property('xalign', 0.5)
        _image.set_property('yalign', 0.0)
        _image.set_from_pixbuf(get_icon('system-search'))
        pix = gtk.gdk.PixbufAnimation(constants.MAIN_ANIMATION)
        _image.set_from_animation(pix)
        
        _hbox.pack_start(_image, expand=False, fill=True, padding=0)
        
        _vbox2 = gtk.VBox()
        _hbox.pack_start(_vbox2, expand=False, fill=True, padding=0)
        
        #Label for progress title
        self._labelTitle = _label = gtk.Label('<big><b>%s</b></big>' % title)
        _label.set_use_markup(True)
        _label.set_alignment(0.0, 0.0)
        _vbox2.pack_start(_label, expand=False, fill=True, padding=0)
        
        #Label for progress description
        self._labelDesc = _label = gtk.Label(description)
        _label.set_line_wrap(True)
        _label.set_alignment(0.5, 0.0)
        _label.set_padding(0, 12)
        _vbox2.pack_start(_label, expand=False, fill=True, padding=0)
        
        _vbox3 = gtk.VBox()
        _vbox.pack_start(_vbox3, False)
        
        self._vbox4 = gtk.VBox()
        _vbox3.pack_start(self._vbox4, False)
        
        #Progress bar widget
        self._progress = _progress = gtk.ProgressBar()
        _vbox3.pack_start(_progress, False)
        
        #text under progressbar
        self._progresstext = _label = gtk.Label('<i>%s</i>' % task)
        _label.set_use_markup(True)
        _label.set_property ("ellipsize", pango.ELLIPSIZE_END)
        _label.set_alignment(0.0, 0.0)
        _vbox3.pack_start(_label, False)
        
        #the cancel button area
        _hbbox =gtk.HButtonBox()
        _hbbox.set_border_width(5)
        _hbbox.set_layout( gtk.BUTTONBOX_END)
        _vbox.pack_start(_hbbox, False)

        #the cancel button, to make users be able to cancel the operation
        self.btnCancel = _btnCancel =gtk.Button(gtk.STOCK_CANCEL)
        _btnCancel.set_use_stock(True)
        _hbbox.pack_start(_btnCancel)
        
        _vbox.show_all()
    
    def __progress_widget(self): return self._progress
    progress_widget = property(fget = __progress_widget, doc = 'Get the gtkProgress widget used in this window.')
        
    def connect_event(self, cancel_hadler):
        """
            This will allow users to set a different 
            function handler to the cancel button click.
            
            @cancel_hadler the function instance that will handler cancel click.
        """
        self.btnCancel.connect("clicked", cancel_hadler)
    
    
    def add_widget(self, widget):
        self._vbox4.pack_start(widget, True, True)
        self._vbox4.show()
        
    def __set_progress_title(self, value): 
            self.set_title(value)
            self._labelTitle.set_markup('<big><b>%s</b></big>' % value)
    progress_title = property(fset= __set_progress_title, doc = 'Set progress window title.')
    
    def __set_progress_description(self, value): self._labelDesc.set_markup(value)
    progress_description = property(fset = __set_progress_description, doc = 'Set progress description text.\n You can use Tango markup tags.')    
    
    def __set_progress_current_task(self, value): 
        self._progresstext.set_markup('<i>%s</i>' % value)
    progress_current_task = property(fset = __set_progress_current_task, doc = 'Set progress current task text under progressbar.')  
    
class ProgressDialog(ProgressDialogBase):
    """
        A progress window to show progress status for long operations.
    """
    def __init__(self, parent = None, start = 0, stop = 0):  
        """
            This will show a progress window  using given parameters..
            
            @parent  is the parent window that controls this progress. Default is None, 
            @start  is the value to start operations, Default is 0. 
            @stop  is the value to mark the end of the progress. Default is 0.
        """
        ProgressDialogBase.__init__(self, parent)
        
        self.starts = start
        self.stop = stop
        self.cancel_status = False
        self.btnCancel.connect("clicked", self.click)
        self.show()
  
    def click(self, *args):
        """
            Handles Cancel button click to hide the progress window.
        """
        self.cancel_status = True
        self.hide()

    def __set_progress_lenght(self, value):
        self.stop = value
        self.update_progress(self.start)
    
    lenght = property( fset =__set_progress_lenght, doc = 'Set progress window lenght.')
    
    def __set_title(self, value):  self.progress_title = value
    title = property(fset = __set_title, doc = 'Set window progress title.')
    
    def __set_description(self, value): self.progress_description = value
    description = property(fset = __set_description, doc = 'Set progress description text.')
    
    def __set_progress_task_text(self, value):
        self.progress_current_task = value
        
    progress_text = property(fset = __set_progress_task_text, doc = 'Set progress current operation text.')
    
    def __can_cancel(self, value = True):
        self.btnCancel.set_sensitive(value)
        self.btnCancel.set_property('visible',value)
    can_cancel_progress = property(fset = __can_cancel, doc = 'Sets if user can cancel the operation. If so, cancel button is visible')
    
    def pulse(self):
        self._progress.pulse()
        self.timer = gobject.timeout_add (100, progress_timeout, self._progress)
    
    def cancel_pulse(self):
        gobject.source_remove(self.timer)
        self._progress.set_fraction(0)
        self._progress.set_text('')
        
    def update_percent(self, value):
        percent = (float(value) / self.stop)
        self._progress.set_text(str(value) + '%' )
        self._progress.set_fraction(percent)
        
    def update_progress(self, value = -1, text = ''):
        """
            Sets a progress bar text and a fraction value.
            if fraction is not set, it will pulse the progressbar.
        """
        if text != '':
            self._progress.set_text(text)
        if value < 0:
            self._progress.pulse()
        else:
            pos = min(max(value, self.starts), self.stop)
            remaining = self.stop - pos
            self._progress.set_text((constants.MESSAGE_0055 % (remaining, self.stop)))
            value = (1.0 - float(remaining) / (self.stop - self.starts))
            self._progress.set_fraction(value)
            
    
