//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/MesocrystalItem.h
//! @brief     Defines class MesocrystalItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_SAMPLE_MESOCRYSTALITEM_H
#define BORNAGAIN_GUI_MODEL_SAMPLE_MESOCRYSTALITEM_H

#include "GUI/Model/Descriptor/SelectionProperty.h"
#include "GUI/Model/Descriptor/VectorProperty.h"
#include "GUI/Model/Sample/FormFactorItemCatalog.h"
#include "GUI/Model/Sample/FormFactorItems.h"
#include "GUI/Model/Sample/ItemWithParticles.h"
#include "GUI/Model/Sample/ItemWithParticlesCatalog.h"
#include "Sample/Lattice/Lattice3D.h"
#include <memory>

class IFormFactor;
class IParticle;
class Mesocrystal;
class MaterialModel;

class MesocrystalItem : public ItemWithParticles {
public:
    explicit MesocrystalItem(const MaterialModel* materials);

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    std::unique_ptr<Mesocrystal> createMesocrystal() const;

    Lattice3D getLattice() const;
    std::unique_ptr<IParticle> getBasis() const;

    std::unique_ptr<IFormFactor> getOuterShape() const;
    template <typename T>
    T* setOuterShapeType();

    SelectionProperty<FormFactorItemCatalog>& outerShapeSelection() { return m_outerShape; }
    void setOuterShape(FormFactorItem* p) { m_outerShape.setCurrentItem(p); }

    ItemWithParticles* basisItem() const { return m_basisParticle.currentItem(); }
    void setBasisItem(ItemWithParticles* basis) { m_basisParticle.setCurrentItem(basis); }

    VectorProperty& vectorA() { return m_vectorA; }
    const VectorProperty& vectorA() const { return m_vectorA; }
    void setVectorA(const R3& v) { m_vectorA.setR3(v); }

    VectorProperty& vectorB() { return m_vectorB; }
    const VectorProperty& vectorB() const { return m_vectorB; }
    void setVectorB(const R3& v) { m_vectorB.setR3(v); }

    VectorProperty& vectorC() { return m_vectorC; }
    const VectorProperty& vectorC() const { return m_vectorC; }
    void setVectorC(const R3& v) { m_vectorC.setR3(v); }

    QVector<ItemWithParticles*> containedItemsWithParticles() const override;

    bool isExpandMesocrystal() const { return m_expandMesocrystal; }
    void setExpandMesocrystal(bool b) { m_expandMesocrystal = b; }

private:
    VectorProperty m_vectorA;
    VectorProperty m_vectorB;
    VectorProperty m_vectorC;
    SelectionProperty<FormFactorItemCatalog> m_outerShape;
    SelectionProperty<ItemWithParticlesCatalog> m_basisParticle;
    const MaterialModel* m_materialModel;
    bool m_expandMesocrystal = true;
};

template <typename T>
T* MesocrystalItem::setOuterShapeType()
{
    auto* p = new T();
    setOuterShape(p);
    return p;
}

#endif // BORNAGAIN_GUI_MODEL_SAMPLE_MESOCRYSTALITEM_H
