//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Instrument/DistributionEditor.h
//! @brief     A widget for selecting a distribution and setting its parameters
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_INSTRUMENT_DISTRIBUTIONEDITOR_H
#define BORNAGAIN_GUI_VIEW_INSTRUMENT_DISTRIBUTIONEDITOR_H

#include "GUI/Support/Data/ID.h"
#include <QGroupBox>
#include <optional>

class BeamDistributionItem;
class DistributionForm;
class DistributionItem;
class DistributionPlot;
class DoubleProperty;
class DoubleSpinBox;
class QComboBox;
class QFormLayout;
class QSpinBox;
class QStackedLayout;
class ScientificSpinBox;


//! configuration to control how the user can enter a mean value
struct MeanConfig {
    bool scientific; //!< whether to use a scientific number notation
};

/// Widget for selecting a distribution (combo box) and input of the corresponding values
/// with respect to the given distribution (e.g. mean and deviation for gauss distribution)

class DistributionSelector : public QWidget {
    Q_OBJECT

public:
    /// \param mean_config controls how the mean value is shown and can be input.
    ///                    If this parameter is \c nullopt then the mean value can not be input
    ///                    and only symmetric distributions are provided
    /// \pre ! mean_config && distributions == Distributions::Symmetric
    ///      (i.e. the combination of all distributions without mean input is currently not
    ///       supported)
    DistributionSelector(std::optional<MeanConfig> mean_config,
                         GUI::ID::Distributions distributions, QWidget* parent,
                         BeamDistributionItem* item, bool allow_distr);

    BeamDistributionItem* item() const;
    GUI::ID::Distributions distributions() const;
    void refresh();

signals:
    void distributionChanged();

private:
    void createDistributionWidgets();
    void createNumSamplesSpinBox(DistributionItem* dist);
    DoubleSpinBox* createSpinBox(DoubleProperty& d);
    void createMeanSpinBox(DoubleProperty& d);

private:
    BeamDistributionItem* m_item;
    std::optional<MeanConfig> m_meanConfig;
    GUI::ID::Distributions m_distributions;
    QFormLayout* m_formLayout;
    QComboBox* m_distributionCombo;
};


/// DistributionSelector as above but contained in a GroupBox with a title and a button
/// to open the distribution dialog

class DistributionEditor : public QGroupBox {
    Q_OBJECT

public:
    DistributionEditor(const QString& title, const std::optional<MeanConfig>& mean_config,
                       GUI::ID::Distributions distributions, QWidget* parent,
                       BeamDistributionItem* item, bool allow_distr = true);

    //! Update UI from data
    void updateData();

signals:
    void distributionChanged();

private:
    void updatePlot();

    DistributionSelector* m_selector;
    DistributionPlot* m_plot;
};

#endif // BORNAGAIN_GUI_VIEW_INSTRUMENT_DISTRIBUTIONEDITOR_H
