import configvalues as conf
from CalLabel import CalLabel
from Observable import Observable
import sfrmapper

import gtk
import pango
import time
import os
import calendar





# this is only here, so that xgettext can find the strings
_MONTHS = [_("January"), _("February"), _("March"),
           _("April"), _("May"), _("June"),
           _("July"), _("August"), _("September"),
           _("October"), _("November"), _("December")]
_WEEKDAYS = [_("Mo"), _("Tu"), _("We"), _("Th"), _("Fr"), _("Sa"), _("Su")]


# widget property for storing the day number
_PROPERTY_DAY = "day"


# make the calendar return English names
calendar.month_name = ["", "January", "February", "March",
                       "April", "May", "June",
                       "July", "August", "September",
                       "October", "November", "December"]
calendar.day_abbr = ["Mo", "Tu", "We", "Th", "Fr", "Sa", "Su"]



#
# Class for a calendar widget.
#
class Cal(gtk.Fixed, Observable):

    OBS_ENTER_DAY = 0
    OBS_LEAVE_DAY = 1
    OBS_OPEN_DAY = 2



    #
    # Constructor.
    #
    def __init__(self):

        self.__month = 1
        self.__year = 1

        self.__first_day = 0
        self.__local_numerals = 0
        self.__have_today_mark = 0
        self.__have_year_coords = 0

        # layout of the day labels: (x, y, width, height, hgap, vgap)
        self.__days_layout = (0, 0, 0, 0, 0, 0)

        # the currently selected day
        self.__selected_day = 0

        # the planner to query
        self.__planner = None
        

        gtk.Fixed.__init__(self)

        # month (and year)
        self.__lbl_month = CalLabel(2, 1, gtk.FALSE)
        self.__lbl_month.show()
        self.put(self.__lbl_month, 0, 0)

        # year alone
        self.__lbl_year = CalLabel(1, 1, gtk.FALSE)
        self.__lbl_year.show()
        self.put(self.__lbl_year, 0, 0)

        # weekdays
        self.__lbl_weekdays = CalLabel(7, 1, gtk.TRUE)
        self.__lbl_weekdays.show()
        self.put(self.__lbl_weekdays, 0, 0)

        # days
        self.__lbl_days = CalLabel(7, 6, gtk.TRUE)
        self.__lbl_days.show()
        self.put(self.__lbl_days, 0, 0)


        # mark for today
        self.__today_mark = gtk.Image()
        self.__today_mark.show()
        self.put(self.__today_mark, -1000, 0)


        #self.connect("button-press-event", self.__on_button_press)
        #self.connect("motion-notify-event", self.__on_day_enter)
        #self.connect("leave-notify-event", self.__on_day_leave)
        self.add_events(gtk.gdk.POINTER_MOTION_MASK)



    #
    # Places the labels.
    #
    def __place_labels(self, config):

        # place month/year
        xpos_month = config.get([conf.SKIN, conf.GEOMETRY, conf.MONTH, conf.X])
        ypos_month = config.get([conf.SKIN, conf.GEOMETRY, conf.MONTH, conf.Y])
        self.move(self.__lbl_month, xpos_month, ypos_month)

        xpos_year = config.get([conf.SKIN, conf.GEOMETRY, conf.YEAR, conf.X])
        ypos_year = config.get([conf.SKIN, conf.GEOMETRY, conf.YEAR, conf.Y])
        self.move(self.__lbl_year, xpos_year, ypos_year)



        # place weekdays
        xpos_wdays = config.get([conf.SKIN, conf.GEOMETRY, conf.WEEKDAYS,
                                     conf.X])
        ypos_wdays = config.get([conf.SKIN, conf.GEOMETRY, conf.WEEKDAYS,
                                     conf.Y])

        self.move(self.__lbl_weekdays, xpos_wdays, ypos_wdays)


        # place days
        xpos_days = config.get([conf.SKIN, conf.GEOMETRY, conf.DAYS, conf.X])
        ypos_days = config.get([conf.SKIN, conf.GEOMETRY, conf.DAYS, conf.Y])

        self.move(self.__lbl_days, xpos_days, ypos_days)



    #
    # Returns the year, month, day from the given coords.
    #
    def __coords2day(self, x, y):

        days_x, days_y, days_width, days_height, \
        days_hgap, days_vgap = self.__days_layout

        if (x < days_x or y < days_y
            or x >= days_x + 7 * (days_width + days_hgap)
            or y >= days_y + 6 * (days_height + days_vgap)):
            return None

        dx = int((x - days_x) / (days_width + days_hgap))
        dy = int((y - days_y) / (days_height + days_vgap))

        lx = days_x + dx * (days_width + days_hgap)
        ly = days_y + dy * (days_height + days_vgap)

        # check if the coords are on a gap
        if (x - lx > days_width or y - ly > days_height): return None

        lbl = self.__lbl_days.get_label_at(lx, ly)
        if (not lbl): return None
        day = lbl.get_data(_PROPERTY_DAY)

        month = self.__month
        year = self.__year

        return (year, month, day)

        


    #
    # Reacts on entering a day label.
    #
    def on_day_enter(self, src, event):

        date = self.__coords2day(event.x, event.y)
        if (not date):
            self.__selected_day = 0
            self.update_observer(self.OBS_LEAVE_DAY)
            return

        year, month, day = date
        if (day == self.__selected_day):
            return
        
        elif (day > 0):
            self.__selected_day = day
            self.update_observer(self.OBS_ENTER_DAY, year, month, day)




    #
    # Reacts on leaving a day label.
    #
    def on_day_leave(self, src, event):

        self.__selected_day = 0
        self.update_observer(self.OBS_LEAVE_DAY)



    #
    # Reacts on pressing a mouse button.
    #
    def on_button_press(self, src, event):

        if (event.button == 1 and event.type == gtk.gdk._2BUTTON_PRESS):
            date = self.__coords2day(event.x, event.y)
            if (date):
                year, month, day = date
                if (day > 0):
                    self.update_observer(self.OBS_OPEN_DAY, year, month, day)
            #end if
        #end if



    #
    # Returns the current day, month and year.
    #
    def __get_current_day(self):

        now = time.localtime(time.time())
        year = now[0]
        month = now[1]
        day = now[2]

        return (day, month, year)



    #
    # Sets the month to be displayed.
    #
    def set_month(self, month = 0, year = 0):

        while (gtk.gdk.events_pending()): gtk.mainiteration()
        gtk.idle_add(self.__idle_set_month, month, year)

    def __idle_set_month(self, month, year):

        curday, curmonth, curyear = self.__get_current_day()
        if (month == 0 or year < 1970):
            month = curmonth; year = curyear

        self.__month = month
        self.__year = year

        # get the calendar sheet
        calendar.setfirstweekday(self.__first_day)
        cal = calendar.month(year, month)
        lines = cal.splitlines()



        monthtext, yeartext = lines[0].split()[:2]
        weekdays = lines[1].split()
        days = "\n".join(lines[2:])

        if (self.__local_numerals):
            yeartext = sfrmapper.map(yeartext, sfrmapper.LOCAL)

        lbl = self.__lbl_month.get_label(0)
        lbl.set_text(_(monthtext) + " ")

        # update the correct year label
        if (self.__have_year_coords):
            lbl = self.__lbl_year.get_label(0)
            lbl.set_text(yeartext + " ")
        else:
            lbl = self.__lbl_month.get_label(1)
            lbl.set_text("\n" + yeartext + " ")


        # update labels
        self.move(self.__today_mark, -1000, 0)

        for i in xrange(7):
            value = weekdays[i]
            lbl = self.__lbl_weekdays.get_label(i)
            lbl.set_text(_(value))


        for i in xrange(42):
            lbl = self.__lbl_days.get_label(i)
            
            value = days[i * 3:i * 3 + 2]
            if (value.strip().isdigit()):
                day = int(value)
            else:
                day = 0

            if (self.__local_numerals):
                value = sfrmapper.map(value, sfrmapper.LOCAL)

            # ask the planner if there are events scheduled for that day
            if (self.__planner.get_events(year, month, day)):
                v = value.strip()
                markup = "<u>" + v + "</u>"
                if (len(v) == 1): markup = " " + markup
                lbl.set_markup(markup)
            else:
                lbl.set_text(value)
                
            lbl.set_data(_PROPERTY_DAY, day)

            # place the mark
            if (self.__have_today_mark
                and month == curmonth and year == curyear and day == curday):
                alloc = lbl.get_allocation()
                mark_alloc = self.__today_mark.get_allocation()
                self.move(self.__today_mark,
                          alloc.x + (alloc.width - mark_alloc.width) / 2,
                          alloc.y + (alloc.height - mark_alloc.height) / 2)
            #end if
        #end for
        self.__lbl_days.update_positions()



    #
    # Returns the currently displayed month and year.
    #
    def get_month(self):

        return (self.__month, self.__year)



    #
    # Scales the given font description from point size to pixel size.
    #
    def __scale_font(self, fontdescription, dpi):

        height = gtk.gdk.screen_height()
        height_inch = height / dpi
        height_pt = height_inch * 72.0
        scale = height_pt / height

        size = fontdescription.get_size()
        size *= scale
        
        fontdescription.set_size(int(size))


    #
    # Sets the planner to use.
    #
    def set_planner(self, planner):

        self.__planner = planner



    #
    # Configures the calendar.
    #
    def set_config(self, config):

        # first day of the week
        self.__first_day = config.get([conf.BEHAVIOUR, conf.FIRST_DAY])

        # whether we use local numerals
        self.__local_numerals = config.get([conf.BEHAVIOUR,
                                            conf.NATIONAL_NUMERALS])

        # resolution
        dpi = config.get([conf.TEMPORARY, conf.DPI])

        # fonts
        font_year = pango.FontDescription(
            config.get([conf.SKIN, conf.FONTS, conf.YEAR]))
        font_month = pango.FontDescription(
            config.get([conf.SKIN, conf.FONTS, conf.MONTH]))
        font_weekdays = pango.FontDescription(
            config.get([conf.SKIN, conf.FONTS, conf.WEEKDAYS]))
        font_days = pango.FontDescription(
            config.get([conf.SKIN, conf.FONTS, conf.DAYS]))

        # colors
        color_month = config.get([conf.SKIN, conf.COLORS, conf.MONTH])
        color_year = config.get([conf.SKIN, conf.COLORS, conf.YEAR])
        color_weekdays = config.get([conf.SKIN, conf.COLORS, conf.WEEKDAYS])
        color_days = config.get([conf.SKIN, conf.COLORS, conf.DAYS])


        # mark for today
        path = config.get([conf.SKIN, conf.PATH])
        self.__today_mark.hide()
        today_mark = config.get([conf.SKIN, conf.GRAPHICS, conf.TODAY])
        if (today_mark):
            self.__today_mark.set_from_file(os.path.join(path, today_mark))
            self.__have_today_mark = 1
        else:
            self.__have_today_mark = 0
        self.__today_mark.show()

        
        self.__scale_font(font_month, dpi)
        self.__scale_font(font_year, dpi)
        self.__scale_font(font_weekdays, dpi)
        self.__scale_font(font_days, dpi)
        

        # geometry
        width_wdays = config.get([conf.SKIN, conf.GEOMETRY, conf.WEEKDAYS,
                                      conf.WIDTH])
        height_wdays = config.get([conf.SKIN, conf.GEOMETRY, conf.WEEKDAYS,
                                       conf.HEIGHT])
        hspace_wdays = config.get([conf.SKIN, conf.GEOMETRY, conf.WEEKDAYS,
                                       conf.HSPACING])

        width_days = config.get([conf.SKIN, conf.GEOMETRY, conf.DAYS,
                                 conf.WIDTH])
        height_days = config.get([conf.SKIN, conf.GEOMETRY, conf.DAYS,
                                  conf.HEIGHT])
        hspace_days = config.get([conf.SKIN, conf.GEOMETRY, conf.DAYS,
                                  conf.HSPACING])
        vspace_days = config.get([conf.SKIN, conf.GEOMETRY, conf.DAYS,
                                  conf.VSPACING])


        # does the year have coords?
        self.__have_year_coords = \
          (config.get([conf.SKIN, conf.GEOMETRY, conf.YEAR, conf.X]) != -1)

        # month and year
        lbl = self.__lbl_month.get_label(0)
        lbl.modify_font(font_month)
        lbl.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse(color_month))

        lbl = self.__lbl_month.get_label(1)
        lbl.set_text("")
        lbl.modify_font(font_year)
        lbl.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse(color_year))

        lbl = self.__lbl_year.get_label(0)
        lbl.set_text("")
        lbl.modify_font(font_year)
        lbl.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse(color_year))



        # weekdays
        self.__lbl_weekdays.set_spacing(hspace_wdays, 0)
        for i in xrange(7):
            lbl = self.__lbl_weekdays.get_label(i)
            lbl.set_size_request(width_wdays, height_wdays)
            lbl.modify_font(font_weekdays)
            lbl.modify_fg(gtk.STATE_NORMAL,
                          gtk.gdk.color_parse(color_weekdays))

        # days
        self.__lbl_days.set_spacing(hspace_days, vspace_days)
        for i in xrange(42):
            lbl = self.__lbl_days.get_label(i)
            lbl.set_size_request(width_days, height_days)
            lbl.modify_font(font_days)
            lbl.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse(color_days))

        self.__days_layout = (config.get([conf.SKIN, conf.GEOMETRY,
                                          conf.DAYS, conf.X]),
                              config.get([conf.SKIN, conf.GEOMETRY,
                                          conf.DAYS, conf.Y]),
                              width_days, height_days,
                              hspace_days, vspace_days)

        self.__place_labels(config)
