package sm3

import (
	"encoding/hex"
	"testing"
)

/* -------------------- Unit tests -----------------------*/
func do_hash(t *testing.T, idx int, input []byte, expected_output [32]byte) {
	var d digest
	var output [32]byte

	d.Init()
	d.Write(input[:])
	copy(output[:], d.Sum(nil))

	if output != expected_output {
		exp := make([]byte, hex.EncodedLen(32))
		hex.Encode(exp, expected_output[:])

		got := make([]byte, hex.EncodedLen(len(output)))
		hex.Encode(got, output[:])

		t.Errorf("[%d]: %s != %s", idx, exp, got)
	}
}

func TestSM3OfficialVectors(t *testing.T) {

	var vectors = []struct {
		in  []byte
		out [32]byte
	}{
		// draft-shen-sm3-hash-01: Example 1
		{
			[]byte("abc"),
			[32]byte{
				0x66, 0xC7, 0xF0, 0xF4, 0x62, 0xEE, 0xED, 0xD9, 0xD1, 0xF2, 0xD4, 0x6B, 0xDC, 0x10, 0xE4, 0xE2,
				0x41, 0x67, 0xC4, 0x87, 0x5C, 0xF2, 0xF7, 0xA2, 0x29, 0x7D, 0xA0, 0x2B, 0x8F, 0x4B, 0xA8, 0xE0,
			},
		},
		// draft-shen-sm3-hash-01: Example 2
		{
			[]byte{
				0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
				0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
				0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
				0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64},
			[32]byte{
				0xde, 0xbe, 0x9f, 0xf9, 0x22, 0x75, 0xb8, 0xa1, 0x38, 0x60, 0x48, 0x89, 0xc1, 0x8e, 0x5a, 0x4d,
				0x6f, 0xdb, 0x70, 0xe5, 0x38, 0x7e, 0x57, 0x65, 0x29, 0x3d, 0xcb, 0xa3, 0x9c, 0x0c, 0x57, 0x32},
		},
		// B.1. GB/T 32918.2-2016 A.2 Example 1
		{
			[]byte{
				0x00, 0x90, 0x41, 0x4C, 0x49, 0x43, 0x45, 0x31, 0x32, 0x33, 0x40, 0x59, 0x41, 0x48, 0x4F, 0x4F,
				0x2E, 0x43, 0x4F, 0x4D, 0x78, 0x79, 0x68, 0xB4, 0xFA, 0x32, 0xC3, 0xFD, 0x24, 0x17, 0x84, 0x2E,
				0x73, 0xBB, 0xFE, 0xFF, 0x2F, 0x3C, 0x84, 0x8B, 0x68, 0x31, 0xD7, 0xE0, 0xEC, 0x65, 0x22, 0x8B,
				0x39, 0x37, 0xE4, 0x98, 0x63, 0xE4, 0xC6, 0xD3, 0xB2, 0x3B, 0x0C, 0x84, 0x9C, 0xF8, 0x42, 0x41,
				0x48, 0x4B, 0xFE, 0x48, 0xF6, 0x1D, 0x59, 0xA5, 0xB1, 0x6B, 0xA0, 0x6E, 0x6E, 0x12, 0xD1, 0xDA,
				0x27, 0xC5, 0x24, 0x9A, 0x42, 0x1D, 0xEB, 0xD6, 0x1B, 0x62, 0xEA, 0xB6, 0x74, 0x64, 0x34, 0xEB,
				0xC3, 0xCC, 0x31, 0x5E, 0x32, 0x22, 0x0B, 0x3B, 0xAD, 0xD5, 0x0B, 0xDC, 0x4C, 0x4E, 0x6C, 0x14,
				0x7F, 0xED, 0xD4, 0x3D, 0x06, 0x80, 0x51, 0x2B, 0xCB, 0xB4, 0x2C, 0x07, 0xD4, 0x73, 0x49, 0xD2,
				0x15, 0x3B, 0x70, 0xC4, 0xE5, 0xD7, 0xFD, 0xFC, 0xBF, 0xA3, 0x6E, 0xA1, 0xA8, 0x58, 0x41, 0xB9,
				0xE4, 0x6E, 0x09, 0xA2, 0x0A, 0xE4, 0xC7, 0x79, 0x8A, 0xA0, 0xF1, 0x19, 0x47, 0x1B, 0xEE, 0x11,
				0x82, 0x5B, 0xE4, 0x62, 0x02, 0xBB, 0x79, 0xE2, 0xA5, 0x84, 0x44, 0x95, 0xE9, 0x7C, 0x04, 0xFF,
				0x4D, 0xF2, 0x54, 0x8A, 0x7C, 0x02, 0x40, 0xF8, 0x8F, 0x1C, 0xD4, 0xE1, 0x63, 0x52, 0xA7, 0x3C,
				0x17, 0xB7, 0xF1, 0x6F, 0x07, 0x35, 0x3E, 0x53, 0xA1, 0x76, 0xD6, 0x84, 0xA9, 0xFE, 0x0C, 0x6B,
				0xB7, 0x98, 0xE8, 0x57,
			},
			[32]byte{
				0xF4, 0xA3, 0x84, 0x89, 0xE3, 0x2B, 0x45, 0xB6, 0xF8, 0x76, 0xE3, 0xAC, 0x21, 0x68, 0xCA, 0x39,
				0x23, 0x62, 0xDC, 0x8F, 0x23, 0x45, 0x9C, 0x1D, 0x11, 0x46, 0xFC, 0x3D, 0xBF, 0xB7, 0xBC, 0x9A,
			},
		},
		//  B.2. GB/T 32918.2-2016 A.2 Example 2
		{
			[]byte{
				0xF4, 0xA3, 0x84, 0x89, 0xE3, 0x2B, 0x45, 0xB6, 0xF8, 0x76, 0xE3, 0xAC, 0x21, 0x68, 0xCA, 0x39,
				0x23, 0x62, 0xDC, 0x8F, 0x23, 0x45, 0x9C, 0x1D, 0x11, 0x46, 0xFC, 0x3D, 0xBF, 0xB7, 0xBC, 0x9A,
				0x6D, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x20, 0x64, 0x69, 0x67, 0x65, 0x73, 0x74,
			},
			[32]byte{
				0xB5, 0x24, 0xF5, 0x52, 0xCD, 0x82, 0xB8, 0xB0, 0x28, 0x47, 0x6E, 0x00, 0x5C, 0x37, 0x7F, 0xB1,
				0x9A, 0x87, 0xE6, 0xFC, 0x68, 0x2D, 0x48, 0xBB, 0x5D, 0x42, 0xE3, 0xD9, 0xB9, 0xEF, 0xFE, 0x76,
			},
		},
		//  B.3. GB/T 32918.2-2016 A.3 Example 1
		{
			[]byte{
				0x00, 0x90, 0x41, 0x4C, 0x49, 0x43, 0x45, 0x31, 0x32, 0x33, 0x40, 0x59, 0x41, 0x48, 0x4F, 0x4F,
				0x2E, 0x43, 0x4F, 0x4D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE7, 0x8B, 0xCD, 0x09, 0x74, 0x6C, 0x20, 0x23, 0x78, 0xA7,
				0xE7, 0x2B, 0x12, 0xBC, 0xE0, 0x02, 0x66, 0xB9, 0x62, 0x7E, 0xCB, 0x0B, 0x5A, 0x25, 0x36, 0x7A,
				0xD1, 0xAD, 0x4C, 0xC6, 0x24, 0x2B, 0x00, 0xCD, 0xB9, 0xCA, 0x7F, 0x1E, 0x6B, 0x04, 0x41, 0xF6,
				0x58, 0x34, 0x3F, 0x4B, 0x10, 0x29, 0x7C, 0x0E, 0xF9, 0xB6, 0x49, 0x10, 0x82, 0x40, 0x0A, 0x62,
				0xE7, 0xA7, 0x48, 0x57, 0x35, 0xFA, 0xDD, 0x01, 0x3D, 0xE7, 0x4D, 0xA6, 0x59, 0x51, 0xC4, 0xD7,
				0x6D, 0xC8, 0x92, 0x20, 0xD5, 0xF7, 0x77, 0x7A, 0x61, 0x1B, 0x1C, 0x38, 0xBA, 0xE2, 0x60, 0xB1,
				0x75, 0x95, 0x1D, 0xC8, 0x06, 0x0C, 0x2B, 0x3E, 0x01, 0x65, 0x96, 0x16, 0x45, 0x28, 0x1A, 0x86,
				0x26, 0x60, 0x7B, 0x91, 0x7F, 0x65, 0x7D, 0x7E, 0x93, 0x82, 0xF1, 0xEA, 0x5C, 0xD9, 0x31, 0xF4,
				0x0F, 0x66, 0x27, 0xF3, 0x57, 0x54, 0x26, 0x53, 0xB2, 0x01, 0x68, 0x65, 0x22, 0x13, 0x0D, 0x59,
				0x0F, 0xB8, 0xDE, 0x63, 0x5D, 0x8F, 0xCA, 0x71, 0x5C, 0xC6, 0xBF, 0x3D, 0x05, 0xBE, 0xF3, 0xF7,
				0x5D, 0xA5, 0xD5, 0x43, 0x45, 0x44, 0x48, 0x16, 0x66, 0x12,
			},
			[32]byte{
				0x26, 0x35, 0x2A, 0xF8, 0x2E, 0xC1, 0x9F, 0x20, 0x7B, 0xBC, 0x6F, 0x94, 0x74, 0xE1, 0x1E, 0x90,
				0xCE, 0x0F, 0x7D, 0xDA, 0xCE, 0x03, 0xB2, 0x7F, 0x80, 0x18, 0x17, 0xE8, 0x97, 0xA8, 0x1F, 0xD5,
			},
		},
		// B.4. GB/T 32918.2-2016 A.3 Example 2
		{
			[]byte{
				0x26, 0x35, 0x2A, 0xF8, 0x2E, 0xC1, 0x9F, 0x20, 0x7B, 0xBC, 0x6F, 0x94, 0x74, 0xE1, 0x1E, 0x90,
				0xCE, 0x0F, 0x7D, 0xDA, 0xCE, 0x03, 0xB2, 0x7F, 0x80, 0x18, 0x17, 0xE8, 0x97, 0xA8, 0x1F, 0xD5,
				0x6D, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x20, 0x64, 0x69, 0x67, 0x65, 0x73, 0x74},
			[32]byte{
				0xAD, 0x67, 0x3C, 0xBD, 0xA3, 0x11, 0x41, 0x71, 0x29, 0xA9, 0xEA, 0xA5, 0xF9, 0xAB, 0x1A, 0xA1,
				0x63, 0x3A, 0xD4, 0x77, 0x18, 0xA8, 0x4D, 0xFD, 0x46, 0xC1, 0x7C, 0x6F, 0xA0, 0xAA, 0x3B, 0x12,
			},
		},
		//  B.5. GB/T 32918.3-2016 A.2 Example 1
		{
			[]byte{
				0x00, 0x90, 0x41, 0x4C, 0x49, 0x43, 0x45, 0x31, 0x32, 0x33, 0x40, 0x59, 0x41, 0x48, 0x4F, 0x4F,
				0x2E, 0x43, 0x4F, 0x4D, 0x78, 0x79, 0x68, 0xB4, 0xFA, 0x32, 0xC3, 0xFD, 0x24, 0x17, 0x84, 0x2E,
				0x73, 0xBB, 0xFE, 0xFF, 0x2F, 0x3C, 0x84, 0x8B, 0x68, 0x31, 0xD7, 0xE0, 0xEC, 0x65, 0x22, 0x8B,
				0x39, 0x37, 0xE4, 0x98, 0x63, 0xE4, 0xC6, 0xD3, 0xB2, 0x3B, 0x0C, 0x84, 0x9C, 0xF8, 0x42, 0x41,
				0x48, 0x4B, 0xFE, 0x48, 0xF6, 0x1D, 0x59, 0xA5, 0xB1, 0x6B, 0xA0, 0x6E, 0x6E, 0x12, 0xD1, 0xDA,
				0x27, 0xC5, 0x24, 0x9A, 0x42, 0x1D, 0xEB, 0xD6, 0x1B, 0x62, 0xEA, 0xB6, 0x74, 0x64, 0x34, 0xEB,
				0xC3, 0xCC, 0x31, 0x5E, 0x32, 0x22, 0x0B, 0x3B, 0xAD, 0xD5, 0x0B, 0xDC, 0x4C, 0x4E, 0x6C, 0x14,
				0x7F, 0xED, 0xD4, 0x3D, 0x06, 0x80, 0x51, 0x2B, 0xCB, 0xB4, 0x2C, 0x07, 0xD4, 0x73, 0x49, 0xD2,
				0x15, 0x3B, 0x70, 0xC4, 0xE5, 0xD7, 0xFD, 0xFC, 0xBF, 0xA3, 0x6E, 0xA1, 0xA8, 0x58, 0x41, 0xB9,
				0xE4, 0x6E, 0x09, 0xA2, 0x30, 0x99, 0x09, 0x3B, 0xF3, 0xC1, 0x37, 0xD8, 0xFC, 0xBB, 0xCD, 0xF4,
				0xA2, 0xAE, 0x50, 0xF3, 0xB0, 0xF2, 0x16, 0xC3, 0x12, 0x2D, 0x79, 0x42, 0x5F, 0xE0, 0x3A, 0x45,
				0xDB, 0xFE, 0x16, 0x55, 0x3D, 0xF7, 0x9E, 0x8D, 0xAC, 0x1C, 0xF0, 0xEC, 0xBA, 0xA2, 0xF2, 0xB4,
				0x9D, 0x51, 0xA4, 0xB3, 0x87, 0xF2, 0xEF, 0xAF, 0x48, 0x23, 0x39, 0x08, 0x6A, 0x27, 0xA8, 0xE0,
				0x5B, 0xAE, 0xD9, 0x8B,
			},
			[32]byte{
				0xE4, 0xD1, 0xD0, 0xC3, 0xCA, 0x4C, 0x7F, 0x11, 0xBC, 0x8F, 0xF8, 0xCB, 0x3F, 0x4C, 0x02, 0xA7,
				0x8F, 0x10, 0x8F, 0xA0, 0x98, 0xE5, 0x1A, 0x66, 0x84, 0x87, 0x24, 0x0F, 0x75, 0xE2, 0x0F, 0x31,
			},
		},
		//  B.6. GB/T 32918.3-2016 A.2 Example 2
		{
			[]byte{
				0x00, 0x88, 0x42, 0x49, 0x4C, 0x4C, 0x34, 0x35, 0x36, 0x40, 0x59, 0x41, 0x48, 0x4F, 0x4F, 0x2E,
				0x43, 0x4F, 0x4D, 0x78, 0x79, 0x68, 0xB4, 0xFA, 0x32, 0xC3, 0xFD, 0x24, 0x17, 0x84, 0x2E, 0x73,
				0xBB, 0xFE, 0xFF, 0x2F, 0x3C, 0x84, 0x8B, 0x68, 0x31, 0xD7, 0xE0, 0xEC, 0x65, 0x22, 0x8B, 0x39,
				0x37, 0xE4, 0x98, 0x63, 0xE4, 0xC6, 0xD3, 0xB2, 0x3B, 0x0C, 0x84, 0x9C, 0xF8, 0x42, 0x41, 0x48,
				0x4B, 0xFE, 0x48, 0xF6, 0x1D, 0x59, 0xA5, 0xB1, 0x6B, 0xA0, 0x6E, 0x6E, 0x12, 0xD1, 0xDA, 0x27,
				0xC5, 0x24, 0x9A, 0x42, 0x1D, 0xEB, 0xD6, 0x1B, 0x62, 0xEA, 0xB6, 0x74, 0x64, 0x34, 0xEB, 0xC3,
				0xCC, 0x31, 0x5E, 0x32, 0x22, 0x0B, 0x3B, 0xAD, 0xD5, 0x0B, 0xDC, 0x4C, 0x4E, 0x6C, 0x14, 0x7F,
				0xED, 0xD4, 0x3D, 0x06, 0x80, 0x51, 0x2B, 0xCB, 0xB4, 0x2C, 0x07, 0xD4, 0x73, 0x49, 0xD2, 0x15,
				0x3B, 0x70, 0xC4, 0xE5, 0xD7, 0xFD, 0xFC, 0xBF, 0xA3, 0x6E, 0xA1, 0xA8, 0x58, 0x41, 0xB9, 0xE4,
				0x6E, 0x09, 0xA2, 0x24, 0x54, 0x93, 0xD4, 0x46, 0xC3, 0x8D, 0x8C, 0xC0, 0xF1, 0x18, 0x37, 0x46,
				0x90, 0xE7, 0xDF, 0x63, 0x3A, 0x8A, 0x4B, 0xFB, 0x33, 0x29, 0xB5, 0xEC, 0xE6, 0x04, 0xB2, 0xB4,
				0xF3, 0x7F, 0x43, 0x53, 0xC0, 0x86, 0x9F, 0x4B, 0x9E, 0x17, 0x77, 0x3D, 0xE6, 0x8F, 0xEC, 0x45,
				0xE1, 0x49, 0x04, 0xE0, 0xDE, 0xA4, 0x5B, 0xF6, 0xCE, 0xCF, 0x99, 0x18, 0xC8, 0x5E, 0xA0, 0x47,
				0xC6, 0x0A, 0x4C,
			},
			[32]byte{
				0x6B, 0x4B, 0x6D, 0x0E, 0x27, 0x66, 0x91, 0xBD, 0x4A, 0x11, 0xBF, 0x72, 0xF4, 0xFB, 0x50, 0x1A,
				0xE3, 0x09, 0xFD, 0xAC, 0xB7, 0x2F, 0xA6, 0xCC, 0x33, 0x6E, 0x66, 0x56, 0x11, 0x9A, 0xBD, 0x67,
			},
		},
		// B.7. GB/T 32918.3-2016 A.2 Example 3
		{
			[]byte{
				0x47, 0xC8, 0x26, 0x53, 0x4D, 0xC2, 0xF6, 0xF1, 0xFB, 0xF2, 0x87, 0x28, 0xDD, 0x65, 0x8F, 0x21,
				0xE1, 0x74, 0xF4, 0x81, 0x79, 0xAC, 0xEF, 0x29, 0x00, 0xF8, 0xB7, 0xF5, 0x66, 0xE4, 0x09, 0x05,
				0xE4, 0xD1, 0xD0, 0xC3, 0xCA, 0x4C, 0x7F, 0x11, 0xBC, 0x8F, 0xF8, 0xCB, 0x3F, 0x4C, 0x02, 0xA7,
				0x8F, 0x10, 0x8F, 0xA0, 0x98, 0xE5, 0x1A, 0x66, 0x84, 0x87, 0x24, 0x0F, 0x75, 0xE2, 0x0F, 0x31,
				0x6B, 0x4B, 0x6D, 0x0E, 0x27, 0x66, 0x91, 0xBD, 0x4A, 0x11, 0xBF, 0x72, 0xF4, 0xFB, 0x50, 0x1A,
				0xE3, 0x09, 0xFD, 0xAC, 0xB7, 0x2F, 0xA6, 0xCC, 0x33, 0x6E, 0x66, 0x56, 0x11, 0x9A, 0xBD, 0x67,
				0x6C, 0xB5, 0x63, 0x38, 0x16, 0xF4, 0xDD, 0x56, 0x0B, 0x1D, 0xEC, 0x45, 0x83, 0x10, 0xCB, 0xCC,
				0x68, 0x56, 0xC0, 0x95, 0x05, 0x32, 0x4A, 0x6D, 0x23, 0x15, 0x0C, 0x40, 0x8F, 0x16, 0x2B, 0xF0,
				0x0D, 0x6F, 0xCF, 0x62, 0xF1, 0x03, 0x6C, 0x0A, 0x1B, 0x6D, 0xAC, 0xCF, 0x57, 0x39, 0x92, 0x23,
				0xA6, 0x5F, 0x7D, 0x7B, 0xF2, 0xD9, 0x63, 0x7E, 0x5B, 0xBB, 0xEB, 0x85, 0x79, 0x61, 0xBF, 0x1A,
				0x17, 0x99, 0xB2, 0xA2, 0xC7, 0x78, 0x29, 0x53, 0x00, 0xD9, 0xA2, 0x32, 0x5C, 0x68, 0x61, 0x29,
				0xB8, 0xF2, 0xB5, 0x33, 0x7B, 0x3D, 0xCF, 0x45, 0x14, 0xE8, 0xBB, 0xC1, 0x9D, 0x90, 0x0E, 0xE5,
				0x54, 0xC9, 0x28, 0x8C, 0x82, 0x73, 0x3E, 0xFD, 0xF7, 0x80, 0x8A, 0xE7, 0xF2, 0x7D, 0x0E, 0x73,
				0x2F, 0x7C, 0x73, 0xA7, 0xD9, 0xAC, 0x98, 0xB7, 0xD8, 0x74, 0x0A, 0x91, 0xD0, 0xDB, 0x3C, 0xF4,
			},
			[32]byte{
				0xFF, 0x49, 0xD9, 0x5B, 0xD4, 0x5F, 0xCE, 0x99, 0xED, 0x54, 0xA8, 0xAD, 0x7A, 0x70, 0x91, 0x10,
				0x9F, 0x51, 0x39, 0x44, 0x42, 0x91, 0x6B, 0xD1, 0x54, 0xD1, 0xDE, 0x43, 0x79, 0xD9, 0x76, 0x47,
			},
		},
		// B.8. GB/T 32918.3-2016 A.2 Example 4
		{
			[]byte{
				0x02, 0x2A, 0xF8, 0x6E, 0xFE, 0x73, 0x2C, 0xF1, 0x2A, 0xD0, 0xE0, 0x9A, 0x1F, 0x25, 0x56, 0xCC,
				0x65, 0x0D, 0x9C, 0xCC, 0xE3, 0xE2, 0x49, 0x86, 0x6B, 0xBB, 0x5C, 0x68, 0x46, 0xA4, 0xC4, 0xA2,
				0x95, 0xFF, 0x49, 0xD9, 0x5B, 0xD4, 0x5F, 0xCE, 0x99, 0xED, 0x54, 0xA8, 0xAD, 0x7A, 0x70, 0x91,
				0x10, 0x9F, 0x51, 0x39, 0x44, 0x42, 0x91, 0x6B, 0xD1, 0x54, 0xD1, 0xDE, 0x43, 0x79, 0xD9, 0x76,
				0x47,
			},
			[32]byte{
				0x28, 0x4C, 0x8F, 0x19, 0x8F, 0x14, 0x1B, 0x50, 0x2E, 0x81, 0x25, 0x0F, 0x15, 0x81, 0xC7, 0xE9,
				0xEE, 0xB4, 0xCA, 0x69, 0x90, 0xF9, 0xE0, 0x2D, 0xF3, 0x88, 0xB4, 0x54, 0x71, 0xF5, 0xBC, 0x5C,
			},
		},
		//  B.9. GB/T 32918.3-2016 A.2 Example 5
		{
			[]byte{
				0x03, 0x2A, 0xF8, 0x6E, 0xFE, 0x73, 0x2C, 0xF1, 0x2A, 0xD0, 0xE0, 0x9A, 0x1F, 0x25, 0x56, 0xCC,
				0x65, 0x0D, 0x9C, 0xCC, 0xE3, 0xE2, 0x49, 0x86, 0x6B, 0xBB, 0x5C, 0x68, 0x46, 0xA4, 0xC4, 0xA2,
				0x95, 0xFF, 0x49, 0xD9, 0x5B, 0xD4, 0x5F, 0xCE, 0x99, 0xED, 0x54, 0xA8, 0xAD, 0x7A, 0x70, 0x91,
				0x10, 0x9F, 0x51, 0x39, 0x44, 0x42, 0x91, 0x6B, 0xD1, 0x54, 0xD1, 0xDE, 0x43, 0x79, 0xD9, 0x76,
				0x47,
			},
			[32]byte{
				0x23, 0x44, 0x4D, 0xAF, 0x8E, 0xD7, 0x53, 0x43, 0x66, 0xCB, 0x90, 0x1C, 0x84, 0xB3, 0xBD, 0xBB,
				0x63, 0x50, 0x4F, 0x40, 0x65, 0xC1, 0x11, 0x6C, 0x91, 0xA4, 0xC0, 0x06, 0x97, 0xE6, 0xCF, 0x7A,
			},
		},
		//  B.11. GB/T 32918.3-2016 A.3 Example 2
		{
			[]byte{
				0x00, 0x88, 0x42, 0x49, 0x4C, 0x4C, 0x34, 0x35, 0x36, 0x40, 0x59, 0x41, 0x48, 0x4F, 0x4F, 0x2E,
				0x43, 0x4F, 0x4D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0xE7, 0x8B, 0xCD, 0x09, 0x74, 0x6C, 0x20, 0x23, 0x78, 0xA7, 0xE7,
				0x2B, 0x12, 0xBC, 0xE0, 0x02, 0x66, 0xB9, 0x62, 0x7E, 0xCB, 0x0B, 0x5A, 0x25, 0x36, 0x7A, 0xD1,
				0xAD, 0x4C, 0xC6, 0x24, 0x2B, 0x00, 0xCD, 0xB9, 0xCA, 0x7F, 0x1E, 0x6B, 0x04, 0x41, 0xF6, 0x58,
				0x34, 0x3F, 0x4B, 0x10, 0x29, 0x7C, 0x0E, 0xF9, 0xB6, 0x49, 0x10, 0x82, 0x40, 0x0A, 0x62, 0xE7,
				0xA7, 0x48, 0x57, 0x35, 0xFA, 0xDD, 0x01, 0x3D, 0xE7, 0x4D, 0xA6, 0x59, 0x51, 0xC4, 0xD7, 0x6D,
				0xC8, 0x92, 0x20, 0xD5, 0xF7, 0x77, 0x7A, 0x61, 0x1B, 0x1C, 0x38, 0xBA, 0xE2, 0x60, 0xB1, 0x75,
				0x95, 0x1D, 0xC8, 0x06, 0x0C, 0x2B, 0x3E, 0x00, 0x34, 0x29, 0x7D, 0xD8, 0x3A, 0xB1, 0x4D, 0x5B,
				0x39, 0x3B, 0x67, 0x12, 0xF3, 0x2B, 0x2F, 0x2E, 0x93, 0x8D, 0x46, 0x90, 0xB0, 0x95, 0x42, 0x4B,
				0x89, 0xDA, 0x88, 0x0C, 0x52, 0xD4, 0xA7, 0xD9, 0x01, 0x99, 0xBB, 0xF1, 0x1A, 0xC9, 0x5A, 0x0E,
				0xA3, 0x4B, 0xBD, 0x00, 0xCA, 0x50, 0xB9, 0x3E, 0xC2, 0x4A, 0xCB, 0x68, 0x33, 0x5D, 0x20, 0xBA,
				0x5D, 0xCF, 0xE3, 0xB3, 0x3B, 0xDB, 0xD2, 0xB6, 0x2D,
			},
			[32]byte{
				0x55, 0x7B, 0xAD, 0x30, 0xE1, 0x83, 0x55, 0x9A, 0xEE, 0xC3, 0xB2, 0x25, 0x6E, 0x1C, 0x7C, 0x11,
				0xF8, 0x70, 0xD2, 0x2B, 0x16, 0x5D, 0x01, 0x5A, 0xCF, 0x94, 0x65, 0xB0, 0x9B, 0x87, 0xB5, 0x27,
			},
		},
		//  B.12. GB/T 32918.3-2016 A.3 Example 3
		{
			[]byte{
				0x00, 0xDA, 0xDD, 0x08, 0x74, 0x06, 0x22, 0x1D, 0x65, 0x7B, 0xC3, 0xFA, 0x79, 0xFF, 0x32, 0x9B,
				0xB0, 0x22, 0xE9, 0xCB, 0x7D, 0xDF, 0xCF, 0xCC, 0xFE, 0x27, 0x7B, 0xE8, 0xCD, 0x4A, 0xE9, 0xB9,
				0x54, 0xEC, 0xF0, 0x08, 0x02, 0x15, 0x97, 0x7B, 0x2E, 0x5D, 0x6D, 0x61, 0xB9, 0x8A, 0x99, 0x44,
				0x2F, 0x03, 0xE8, 0x80, 0x3D, 0xC3, 0x9E, 0x34, 0x9F, 0x8D, 0xCA, 0x56, 0x21, 0xA9, 0xAC, 0xDF,
				0x2B, 0x55, 0x7B, 0xAD, 0x30, 0xE1, 0x83, 0x55, 0x9A, 0xEE, 0xC3, 0xB2, 0x25, 0x6E, 0x1C, 0x7C,
				0x11, 0xF8, 0x70, 0xD2, 0x2B, 0x16, 0x5D, 0x01, 0x5A, 0xCF, 0x94, 0x65, 0xB0, 0x9B, 0x87, 0xB5,
				0x27, 0x01, 0x81, 0x07, 0x65, 0x43, 0xED, 0x19, 0x05, 0x8C, 0x38, 0xB3, 0x13, 0xD7, 0x39, 0x92,
				0x1D, 0x46, 0xB8, 0x00, 0x94, 0xD9, 0x61, 0xA1, 0x36, 0x73, 0xD4, 0xA5, 0xCF, 0x8C, 0x71, 0x59,
				0xE3, 0x04, 0x01, 0xD8, 0xCF, 0xFF, 0x7C, 0xA2, 0x7A, 0x01, 0xA2, 0xE8, 0x8C, 0x18, 0x67, 0x37,
				0x48, 0xFD, 0xE9, 0xA7, 0x4C, 0x1F, 0x9B, 0x45, 0x64, 0x6E, 0xCA, 0x09, 0x97, 0x29, 0x3C, 0x15,
				0xC3, 0x4D, 0xD8, 0x00, 0x2A, 0x48, 0x32, 0xB4, 0xDC, 0xD3, 0x99, 0xBA, 0xAB, 0x3F, 0xFF, 0xE7,
				0xDD, 0x6C, 0xE6, 0xED, 0x68, 0xCC, 0x43, 0xFF, 0xA5, 0xF2, 0x62, 0x3B, 0x9B, 0xD0, 0x4E, 0x46,
				0x8D, 0x32, 0x2A, 0x2A, 0x00, 0x16, 0x59, 0x9B, 0xB5, 0x2E, 0xD9, 0xEA, 0xFA, 0xD0, 0x1C, 0xFA,
				0x45, 0x3C, 0xF3, 0x05, 0x2E, 0xD6, 0x01, 0x84, 0xD2, 0xEE, 0xCF, 0xD4, 0x2B, 0x52, 0xDB, 0x74,
				0x11, 0x0B, 0x98, 0x4C, 0x23,
			},
			[32]byte{
				0xE0, 0x5F, 0xE2, 0x87, 0xB7, 0x3B, 0x0C, 0xE6, 0x63, 0x95, 0x24, 0xCD, 0x86, 0x69, 0x43, 0x11,
				0x56, 0x29, 0x14, 0xF4, 0xF6, 0xA3, 0x42, 0x41, 0x01, 0xD8, 0x85, 0xF8, 0x8B, 0x05, 0x36, 0x9C,
			},
		},
		// B.13. GB/T 32918.3-2016 A.3 Example 4
		{
			[]byte{
				0x02, 0x01, 0xF0, 0x46, 0x4B, 0x1E, 0x81, 0x68, 0x4E, 0x5E, 0xD6, 0xEF, 0x28, 0x1B, 0x55, 0x62,
				0x4E, 0xF4, 0x6C, 0xAA, 0x3B, 0x2D, 0x37, 0x48, 0x43, 0x72, 0xD9, 0x16, 0x10, 0xB6, 0x98, 0x25,
				0x2C, 0xC9, 0xE0, 0x5F, 0xE2, 0x87, 0xB7, 0x3B, 0x0C, 0xE6, 0x63, 0x95, 0x24, 0xCD, 0x86, 0x69,
				0x43, 0x11, 0x56, 0x29, 0x14, 0xF4, 0xF6, 0xA3, 0x42, 0x41, 0x01, 0xD8, 0x85, 0xF8, 0x8B, 0x05,
				0x36, 0x9C,
			},
			[32]byte{
				0x4E, 0xB4, 0x7D, 0x28, 0xAD, 0x39, 0x06, 0xD6, 0x24, 0x4D, 0x01, 0xE0, 0xF6, 0xAE, 0xC7, 0x3B,
				0x0B, 0x51, 0xDE, 0x15, 0x74, 0xC1, 0x37, 0x98, 0x18, 0x4E, 0x48, 0x33, 0xDB, 0xAE, 0x29, 0x5A,
			},
		},
		//  B.14. GB/T 32918.3-2016 A.3 Example 5
		{
			[]byte{
				0x03, 0x01, 0xF0, 0x46, 0x4B, 0x1E, 0x81, 0x68, 0x4E, 0x5E, 0xD6, 0xEF, 0x28, 0x1B, 0x55, 0x62,
				0x4E, 0xF4, 0x6C, 0xAA, 0x3B, 0x2D, 0x37, 0x48, 0x43, 0x72, 0xD9, 0x16, 0x10, 0xB6, 0x98, 0x25,
				0x2C, 0xC9, 0xE0, 0x5F, 0xE2, 0x87, 0xB7, 0x3B, 0x0C, 0xE6, 0x63, 0x95, 0x24, 0xCD, 0x86, 0x69,
				0x43, 0x11, 0x56, 0x29, 0x14, 0xF4, 0xF6, 0xA3, 0x42, 0x41, 0x01, 0xD8, 0x85, 0xF8, 0x8B, 0x05,
				0x36, 0x9C,
			},
			[32]byte{
				0x58, 0x8A, 0xA6, 0x70, 0x64, 0xF2, 0x4D, 0xC2, 0x7C, 0xCA, 0xA1, 0xFA, 0xB7, 0xE2, 0x7D, 0xFF,
				0x81, 0x1D, 0x50, 0x0A, 0xD7, 0xEF, 0x2F, 0xB8, 0xF6, 0x9D, 0xDF, 0x48, 0xCC, 0x0F, 0xEC, 0xB7,
			},
		},
		// B.15. GB/T 32918.4-2016 A.2 Example 1
		{
			[]byte{
				0x57, 0xE7, 0xB6, 0x36, 0x23, 0xFA, 0xE5, 0xF0, 0x8C, 0xDA, 0x46, 0x8E, 0x87, 0x2A, 0x20, 0xAF,
				0xA0, 0x3D, 0xED, 0x41, 0xBF, 0x14, 0x03, 0x77, 0x65, 0x6E, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69,
				0x6F, 0x6E, 0x20, 0x73, 0x74, 0x61, 0x6E, 0x64, 0x61, 0x72, 0x64, 0x0E, 0x04, 0x0D, 0xC8, 0x3A,
				0xF3, 0x1A, 0x67, 0x99, 0x1F, 0x2B, 0x01, 0xEB, 0xF9, 0xEF, 0xD8, 0x88, 0x1F, 0x0A, 0x04, 0x93,
				0x00, 0x06, 0x03,
			},
			[32]byte{
				0x6A, 0xFB, 0x3B, 0xCE, 0xBD, 0x76, 0xF8, 0x2B, 0x25, 0x2C, 0xE5, 0xEB, 0x25, 0xB5, 0x79, 0x96,
				0x86, 0x90, 0x2B, 0x8C, 0xF2, 0xFD, 0x87, 0x53, 0x6E, 0x55, 0xEF, 0x76, 0x03, 0xB0, 0x9E, 0x7C,
			},
		},
		//  B.16. GB/T 32918.4-2016 A.2 Example 2
		{
			[]byte{
				0x64, 0xD2, 0x0D, 0x27, 0xD0, 0x63, 0x29, 0x57, 0xF8, 0x02, 0x8C, 0x1E, 0x02, 0x4F, 0x6B, 0x02,
				0xED, 0xF2, 0x31, 0x02, 0xA5, 0x66, 0xC9, 0x32, 0xAE, 0x8B, 0xD6, 0x13, 0xA8, 0xE8, 0x65, 0xFE,
				0x65, 0x6E, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x73, 0x74, 0x61, 0x6E, 0x64,
				0x61, 0x72, 0x64, 0x58, 0xD2, 0x25, 0xEC, 0xA7, 0x84, 0xAE, 0x30, 0x0A, 0x81, 0xA2, 0xD4, 0x82,
				0x81, 0xA8, 0x28, 0xE1, 0xCE, 0xDF, 0x11, 0xC4, 0x21, 0x90, 0x99, 0x84, 0x02, 0x65, 0x37, 0x50,
				0x77, 0xBF, 0x78,
			},
			[32]byte{
				0x9C, 0x3D, 0x73, 0x60, 0xC3, 0x01, 0x56, 0xFA, 0xB7, 0xC8, 0x0A, 0x02, 0x76, 0x71, 0x2D, 0xA9,
				0xD8, 0x09, 0x4A, 0x63, 0x4B, 0x76, 0x6D, 0x3A, 0x28, 0x5E, 0x07, 0x48, 0x06, 0x53, 0x42, 0x6D,
			},
		},
		//  B.17. GB/T 32918.4-2016 A.3 Example 1
		{
			[]byte{
				0x64, 0xD2, 0x0D, 0x27, 0xD0, 0x63, 0x29, 0x57, 0xF8, 0x02, 0x8C, 0x1E, 0x02, 0x4F, 0x6B, 0x02,
				0xED, 0xF2, 0x31, 0x02, 0xA5, 0x66, 0xC9, 0x32, 0xAE, 0x8B, 0xD6, 0x13, 0xA8, 0xE8, 0x65, 0xFE,
				0x65, 0x6E, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x73, 0x74, 0x61, 0x6E, 0x64,
				0x61, 0x72, 0x64, 0x58, 0xD2, 0x25, 0xEC, 0xA7, 0x84, 0xAE, 0x30, 0x0A, 0x81, 0xA2, 0xD4, 0x82,
				0x81, 0xA8, 0x28, 0xE1, 0xCE, 0xDF, 0x11, 0xC4, 0x21, 0x90, 0x99, 0x84, 0x02, 0x65, 0x37, 0x50,
				0x77, 0xBF, 0x78,
			},
			[32]byte{
				0x9C, 0x3D, 0x73, 0x60, 0xC3, 0x01, 0x56, 0xFA, 0xB7, 0xC8, 0x0A, 0x02, 0x76, 0x71, 0x2D, 0xA9,
				0xD8, 0x09, 0x4A, 0x63, 0x4B, 0x76, 0x6D, 0x3A, 0x28, 0x5E, 0x07, 0x48, 0x06, 0x53, 0x42, 0x6D,
			},
		},
		//  B.18. GB/T 32918.4-2016 A.3 Example 2
		{
			[]byte{
				0x00, 0x83, 0xE6, 0x28, 0xCF, 0x70, 0x1E, 0xE3, 0x14, 0x1E, 0x88, 0x73, 0xFE, 0x55, 0x93, 0x6A,
				0xDF, 0x24, 0x96, 0x3F, 0x5D, 0xC9, 0xC6, 0x48, 0x05, 0x66, 0xC8, 0x0F, 0x8A, 0x1D, 0x8C, 0xC5,
				0x1B, 0x65, 0x6E, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x73, 0x74, 0x61, 0x6E,
				0x64, 0x61, 0x72, 0x64, 0x01, 0x52, 0x4C, 0x64, 0x7F, 0x0C, 0x04, 0x12, 0xDE, 0xFD, 0x46, 0x8B,
				0xDA, 0x3A, 0xE0, 0xE5, 0xA8, 0x0F, 0xCC, 0x8F, 0x5C, 0x99, 0x0F, 0xEE, 0x11, 0x60, 0x29, 0x29,
				0x23, 0x2D, 0xCD, 0x9F, 0x36,
			},
			[32]byte{
				0x73, 0xA4, 0x86, 0x25, 0xD3, 0x75, 0x8F, 0xA3, 0x7B, 0x3E, 0xAB, 0x80, 0xE9, 0xCF, 0xCA, 0xBA,
				0x66, 0x5E, 0x31, 0x99, 0xEA, 0x15, 0xA1, 0xFA, 0x81, 0x89, 0xD9, 0x6F, 0x57, 0x91, 0x25, 0xE4,
			},
		},
	}

	for i := range vectors {
		do_hash(t, i, vectors[i].in[:], vectors[i].out)
	}
}

func TestSM3Sizes(t *testing.T) {
	/*
	 * Tests KATs for input of all zeros and length from 0 to 63
	 * DISCLAIMER:
	 * Those vectors come from "SM3 tiny" (https://github.com/odzhan/sm3.git)
	 * and they do depend on correctness of "SM3 tiny" implementation
	 */
	kat := [][32]byte{
		{0x1a, 0xb2, 0x1d, 0x83, 0x55, 0xcf, 0xa1, 0x7f, 0x8e, 0x61, 0x19, 0x48, 0x31, 0xe8, 0x1a, 0x8f,
			0x22, 0xbe, 0xc8, 0xc7, 0x28, 0xfe, 0xfb, 0x74, 0x7e, 0xd0, 0x35, 0xeb, 0x50, 0x82, 0xaa, 0x2b},
		{0x2d, 0xae, 0xf6, 0x0e, 0x7a, 0x0b, 0x8f, 0x5e, 0x02, 0x4c, 0x81, 0xcd, 0x2a, 0xb3, 0x10, 0x9f,
			0x2b, 0x4f, 0x15, 0x5c, 0xf8, 0x3a, 0xde, 0xb2, 0xae, 0x55, 0x32, 0xf7, 0x4a, 0x15, 0x7f, 0xdf},
		{0xaf, 0x83, 0xa9, 0x66, 0x22, 0x20, 0x57, 0xac, 0x76, 0x12, 0x46, 0xa7, 0x54, 0x3c, 0x58, 0x0d,
			0x91, 0x11, 0x01, 0x4f, 0x4e, 0x5e, 0x3c, 0xb1, 0x28, 0x1d, 0xb3, 0x31, 0x51, 0x16, 0x03, 0x35},
		{0x18, 0x39, 0x20, 0xf0, 0x0e, 0x15, 0xa0, 0x43, 0x3e, 0xe3, 0xa8, 0xfc, 0x90, 0xdd, 0x9a, 0xc1,
			0x64, 0xc4, 0x14, 0x2c, 0xcf, 0x63, 0xca, 0x18, 0x9a, 0x8f, 0x64, 0x5e, 0xc9, 0x6f, 0xf8, 0xde},
		{0xaf, 0xcc, 0x87, 0x0f, 0xa2, 0x0c, 0x50, 0x79, 0x95, 0x49, 0x97, 0x94, 0x37, 0x1e, 0x8c, 0x25,
			0xe3, 0xa7, 0x31, 0x0f, 0xa7, 0x22, 0x00, 0xc1, 0x09, 0x37, 0x99, 0x73, 0xae, 0x23, 0x68, 0x45},
		{0x04, 0x71, 0x33, 0x17, 0xea, 0x89, 0x4e, 0x89, 0x0d, 0x27, 0xfb, 0x1c, 0xf0, 0x80, 0x0e, 0xad,
			0x81, 0x83, 0xf6, 0x47, 0xfd, 0x2f, 0xd3, 0x11, 0x2d, 0x15, 0x1f, 0x8a, 0xc2, 0x69, 0x61, 0x7c},
		{0x40, 0x7b, 0x8b, 0x70, 0x20, 0x99, 0x35, 0x66, 0xd8, 0x15, 0x36, 0x31, 0xa6, 0x7c, 0xe5, 0x83,
			0x66, 0x47, 0xe6, 0xce, 0xce, 0x0d, 0x61, 0x8c, 0x7b, 0x86, 0x7d, 0x41, 0x6a, 0x52, 0x95, 0xc5},
		{0xfa, 0x35, 0x64, 0xff, 0x6f, 0x77, 0x6a, 0x9c, 0xfc, 0x53, 0xa1, 0xf9, 0xb8, 0x6b, 0xd0, 0xdc,
			0x74, 0x7a, 0x47, 0x31, 0xa4, 0x49, 0x11, 0x6e, 0x45, 0xc0, 0xad, 0x0d, 0x05, 0xf5, 0x64, 0xb7},
		{0xb7, 0x86, 0x91, 0x08, 0xf1, 0x51, 0xba, 0x70, 0x67, 0x2e, 0x10, 0x15, 0x3c, 0xc9, 0x0f, 0xc2,
			0xb5, 0xd4, 0x50, 0x6a, 0x96, 0x3f, 0x03, 0xb6, 0xc5, 0xb4, 0xaa, 0x3b, 0xce, 0xa0, 0x9d, 0x83},
		{0xe5, 0xef, 0xbb, 0x13, 0x32, 0x53, 0xd8, 0x86, 0x99, 0x1e, 0xa6, 0x0b, 0x55, 0xc1, 0x3f, 0xb2,
			0x60, 0x4e, 0xfc, 0x76, 0x3e, 0x72, 0x99, 0xfc, 0x58, 0xb4, 0x41, 0xb0, 0x07, 0xaf, 0xdf, 0xa2},
		{0xf0, 0xd0, 0x1a, 0x14, 0xea, 0xf1, 0x40, 0xce, 0xc6, 0xe3, 0xc9, 0x0e, 0x71, 0x46, 0x98, 0xc9,
			0xce, 0xb0, 0x3c, 0xbb, 0x93, 0x33, 0x0c, 0x70, 0xc2, 0xa3, 0x2f, 0x58, 0x7b, 0x4c, 0x8f, 0x11},
		{0x5e, 0x02, 0xae, 0x6a, 0x90, 0x61, 0x13, 0x84, 0x0d, 0x79, 0xea, 0x81, 0x6b, 0xf8, 0xac, 0x73,
			0x2b, 0xbe, 0xe9, 0x98, 0x01, 0x7a, 0xfd, 0x3e, 0x34, 0xa4, 0xf0, 0xd0, 0x33, 0x6e, 0xf2, 0x0a},
		{0xf2, 0x4b, 0xa4, 0xfa, 0x51, 0x5b, 0xc2, 0xce, 0x76, 0x91, 0xf2, 0x47, 0x91, 0x05, 0xe5, 0x7e,
			0xa2, 0x22, 0xf9, 0x36, 0xd2, 0xdc, 0x07, 0x22, 0xf3, 0xf4, 0x4a, 0x98, 0x45, 0xbb, 0xf6, 0x69},
		{0xa3, 0x69, 0x3b, 0x02, 0xb1, 0x87, 0xb4, 0xa0, 0xa0, 0x34, 0x9a, 0x1b, 0x91, 0x13, 0x2f, 0x5e,
			0x24, 0x68, 0xba, 0xdb, 0xad, 0x0f, 0x37, 0x4e, 0x85, 0x99, 0xe4, 0x0e, 0x49, 0xcf, 0x58, 0x56},
		{0x34, 0x4f, 0x85, 0x14, 0x71, 0x10, 0x5f, 0x50, 0x9b, 0x6b, 0x24, 0xc1, 0x8b, 0x92, 0x61, 0xfb,
			0x7f, 0x8c, 0xe6, 0xb1, 0x70, 0x74, 0x12, 0xf3, 0xe6, 0x35, 0xab, 0x26, 0x4a, 0xfb, 0xef, 0x15},
		{0x41, 0xc8, 0xe1, 0xb9, 0x5b, 0xdb, 0xb5, 0xb0, 0x55, 0x1d, 0xfc, 0x44, 0x42, 0x37, 0xd9, 0x06,
			0x59, 0xc8, 0x93, 0x06, 0x93, 0x8b, 0x87, 0xb5, 0xc9, 0x85, 0xca, 0x75, 0xf5, 0xf4, 0x75, 0x5c},
		{0x10, 0x6e, 0x34, 0xa2, 0xb8, 0xc7, 0xbb, 0x13, 0x15, 0x6c, 0xfd, 0xd0, 0xd9, 0x13, 0x79, 0xdc,
			0xc4, 0x75, 0x43, 0xdc, 0xf9, 0x78, 0x7c, 0x68, 0xae, 0x5e, 0xb5, 0x82, 0x62, 0x0a, 0xe6, 0xe8},
		{0xbf, 0xc0, 0xa0, 0x7b, 0xd1, 0xa6, 0xf9, 0x50, 0xba, 0x90, 0x0b, 0x93, 0xce, 0x4d, 0xbd, 0x5b,
			0x3a, 0x79, 0x05, 0x49, 0x80, 0xc4, 0xe6, 0x5b, 0x80, 0xb0, 0x1b, 0x7c, 0x18, 0xff, 0xbb, 0x7f},
		{0x34, 0x28, 0xc2, 0xa2, 0x6d, 0x9d, 0xf6, 0x5d, 0x91, 0xe6, 0xbe, 0x9c, 0xe6, 0x0b, 0x19, 0x33,
			0x79, 0x03, 0x8b, 0x13, 0x16, 0xc1, 0x1e, 0xfc, 0x83, 0x8c, 0x47, 0xc4, 0xb1, 0xc2, 0x7f, 0x01},
		{0x12, 0x3b, 0x5b, 0x7e, 0xb8, 0x60, 0x04, 0x27, 0x6c, 0x4e, 0xa6, 0x4f, 0x15, 0xa8, 0x0b, 0x23,
			0x4f, 0xdb, 0xa4, 0xcc, 0x4b, 0x25, 0x6a, 0x8c, 0xbc, 0xba, 0x21, 0x4b, 0xd5, 0x2e, 0x7f, 0x52},
		{0x18, 0x85, 0x08, 0x04, 0xc9, 0x2e, 0x97, 0xe1, 0x0d, 0xae, 0xac, 0xd8, 0xdd, 0xa0, 0x02, 0x0b,
			0xbd, 0x69, 0x6a, 0xf1, 0xf1, 0x9e, 0x1e, 0xe6, 0xdc, 0x19, 0xfa, 0x2c, 0xfc, 0x9b, 0x34, 0xd6},
		{0x7a, 0x5e, 0xc2, 0x30, 0x95, 0x16, 0x83, 0x2b, 0x8b, 0x42, 0x87, 0x86, 0x15, 0x11, 0xb8, 0x39,
			0x29, 0x20, 0xe2, 0x47, 0x00, 0xcc, 0x5f, 0x17, 0xe2, 0x2b, 0x4f, 0x51, 0x3d, 0x72, 0xdb, 0x15},
		{0x00, 0x54, 0x01, 0xf5, 0x8f, 0x96, 0xd3, 0xdd, 0x7a, 0xde, 0x6a, 0xf5, 0x56, 0x5f, 0xa0, 0x4b,
			0xc8, 0x8b, 0xc4, 0x65, 0xe7, 0x5c, 0x83, 0xb7, 0x74, 0xd4, 0xad, 0x3a, 0x1c, 0xb5, 0xdf, 0xf3},
		{0x37, 0x4b, 0x03, 0xae, 0x83, 0xdf, 0x35, 0x37, 0x53, 0x12, 0x8d, 0x72, 0x37, 0xcc, 0x68, 0xf9,
			0xc0, 0x36, 0xf4, 0x2d, 0x91, 0xcf, 0x02, 0xa1, 0x0b, 0x4f, 0x3d, 0x5a, 0x7e, 0xa6, 0x11, 0x94},
		{0xfb, 0x0f, 0x7c, 0xaf, 0xbe, 0x9a, 0x6c, 0x0e, 0x2c, 0x6c, 0xb8, 0xcf, 0x75, 0xe3, 0x4f, 0x6d,
			0x44, 0x4d, 0x85, 0xee, 0x7d, 0x9e, 0x46, 0x07, 0xea, 0x8b, 0xb0, 0x2d, 0x2d, 0xa6, 0xaa, 0x8f},
		{0x90, 0x92, 0x84, 0xa6, 0x98, 0x30, 0xfa, 0x75, 0xc2, 0x51, 0x16, 0x3a, 0xd0, 0x98, 0xc6, 0x39,
			0xf5, 0x16, 0x42, 0x52, 0x43, 0x92, 0x11, 0x1e, 0x76, 0xfb, 0x4f, 0x09, 0xb6, 0xd8, 0x31, 0xcc},
		{0x30, 0x0c, 0xf9, 0x66, 0xaf, 0x7e, 0x14, 0x1a, 0xa2, 0x83, 0xa8, 0x97, 0xc9, 0x18, 0xe5, 0xcd,
			0x7b, 0x88, 0x0b, 0x6c, 0x9f, 0xe5, 0x8f, 0xab, 0x94, 0xde, 0x0b, 0x44, 0x9a, 0x70, 0x9a, 0x83},
		{0x8f, 0xd0, 0x5c, 0x6b, 0x9c, 0x2a, 0xab, 0x73, 0xf5, 0xc3, 0xde, 0x11, 0xd0, 0xfb, 0x79, 0x28,
			0x50, 0xcd, 0x81, 0xa9, 0xd9, 0x3a, 0xdc, 0xdb, 0x7b, 0x8c, 0x86, 0x0a, 0x75, 0x65, 0x0e, 0xac},
		{0x33, 0x6d, 0x8b, 0x57, 0x9a, 0x20, 0x80, 0x57, 0x33, 0xfc, 0x48, 0x4a, 0x65, 0xaa, 0x9e, 0xca,
			0x6d, 0x51, 0x2f, 0xd8, 0x9d, 0x38, 0x1f, 0xfc, 0xaf, 0xd7, 0x21, 0x07, 0x3b, 0xae, 0x34, 0xbd},
		{0x88, 0xc1, 0xf6, 0xf2, 0x8f, 0xe2, 0xa3, 0xa9, 0xf9, 0x47, 0x9d, 0x2e, 0xd8, 0xff, 0x2c, 0xeb,
			0x3b, 0x26, 0x0b, 0x1d, 0x91, 0xa4, 0x7f, 0xc3, 0xaf, 0x00, 0x10, 0x9a, 0xca, 0x4e, 0xdb, 0x52},
		{0x46, 0xcc, 0x01, 0xc6, 0xef, 0xc0, 0x31, 0xe0, 0x4c, 0x9b, 0x49, 0x0a, 0x3f, 0xb6, 0x0c, 0x2b,
			0x7d, 0x27, 0x3a, 0x35, 0xb0, 0x03, 0xfd, 0x1f, 0x3e, 0x22, 0x7b, 0xc5, 0x8e, 0x88, 0x6c, 0x54},
		{0x12, 0x19, 0x18, 0x39, 0xe9, 0x91, 0x63, 0xc2, 0x11, 0xa0, 0xfc, 0x05, 0x14, 0x85, 0xe3, 0xa3,
			0x44, 0x43, 0x75, 0x85, 0x29, 0xd3, 0x8e, 0x93, 0xc6, 0xb7, 0x92, 0x6b, 0x82, 0xf2, 0x6d, 0x89},
		{0xe0, 0xba, 0xb8, 0xf4, 0xd8, 0x17, 0x2b, 0xa2, 0x45, 0x19, 0x0d, 0x13, 0xc9, 0x41, 0x17, 0xe9,
			0x3b, 0x82, 0x16, 0x6c, 0x25, 0xb2, 0xb6, 0x98, 0x83, 0x35, 0x0c, 0x19, 0x2c, 0x90, 0x51, 0x40},
		{0x8f, 0xa0, 0xdd, 0x2c, 0x79, 0x8c, 0x5b, 0xba, 0xda, 0x41, 0x5e, 0x5d, 0x11, 0x67, 0xa3, 0x70,
			0x72, 0xee, 0x65, 0x55, 0x8a, 0x2c, 0xea, 0x40, 0x8b, 0xc5, 0x21, 0x4c, 0x9f, 0xd2, 0x8f, 0x3c},
		{0x49, 0xcb, 0x1d, 0xa7, 0x13, 0xc9, 0x6d, 0xbc, 0xc6, 0xae, 0x4e, 0xb3, 0x42, 0x2a, 0x8c, 0x5d,
			0xc8, 0xf1, 0x89, 0x76, 0x78, 0x2f, 0xcd, 0x6c, 0xbc, 0xfc, 0x03, 0x0c, 0x83, 0x54, 0x9e, 0x57},
		{0x78, 0x54, 0x9d, 0xab, 0xeb, 0xaa, 0x14, 0xde, 0xe4, 0x9e, 0x98, 0xf0, 0xeb, 0x01, 0xdc, 0xb6,
			0xda, 0x5f, 0xb0, 0x61, 0xdd, 0x6d, 0xb1, 0xf2, 0x1f, 0x8f, 0xe4, 0xfb, 0xd7, 0x5b, 0x10, 0x29},
		{0xee, 0x9d, 0x28, 0xda, 0xd8, 0x81, 0xd7, 0xf5, 0x91, 0x89, 0x9b, 0x4d, 0x10, 0x5e, 0x38, 0x5b,
			0x19, 0x90, 0x11, 0x1f, 0x62, 0x46, 0x34, 0xd3, 0xe1, 0xd5, 0x81, 0x17, 0x7c, 0x2f, 0x04, 0xb0},
		{0xc3, 0x8a, 0xd8, 0x54, 0x46, 0x46, 0x7c, 0x23, 0x67, 0x07, 0xb2, 0x37, 0xf3, 0xc4, 0x93, 0x34,
			0x17, 0x68, 0x84, 0xe5, 0x91, 0x34, 0xce, 0x52, 0x30, 0xd5, 0xbe, 0xb8, 0x4c, 0xf6, 0x59, 0x29},
		{0xc2, 0x85, 0x37, 0xce, 0x34, 0xb0, 0xa1, 0xc9, 0x39, 0xf1, 0xfe, 0xba, 0xcb, 0x14, 0x96, 0xc7,
			0xfe, 0xeb, 0xa7, 0x58, 0xfc, 0x86, 0x77, 0x31, 0x3b, 0x98, 0x66, 0xbd, 0xab, 0x42, 0xd2, 0x28},
		{0x70, 0x75, 0x94, 0x1a, 0xe1, 0x78, 0x0f, 0x31, 0xb0, 0xf8, 0xf2, 0x30, 0x82, 0x46, 0x81, 0xa8,
			0xc7, 0x1a, 0x91, 0x67, 0xb1, 0x29, 0x67, 0xc0, 0xd3, 0xbf, 0x64, 0xff, 0x1e, 0xa6, 0x37, 0xed},
		{0x79, 0xf6, 0x22, 0x22, 0xb7, 0xf9, 0xd0, 0xeb, 0x5c, 0xbd, 0x44, 0x59, 0xfc, 0xd4, 0x7f, 0x6d,
			0x24, 0x40, 0x03, 0xd4, 0x16, 0x42, 0x7b, 0x06, 0x96, 0x56, 0x07, 0x72, 0xb0, 0x5a, 0x73, 0x59},
		{0x2a, 0xea, 0x4b, 0x1d, 0x0c, 0x2b, 0x92, 0x15, 0xbf, 0x36, 0xed, 0x1c, 0x6f, 0xc8, 0x40, 0x5c,
			0x86, 0x4a, 0x51, 0x5e, 0x40, 0x55, 0x93, 0x56, 0xa9, 0x84, 0x2a, 0xb6, 0x8c, 0xcd, 0x7e, 0x13},
		{0xaf, 0x4d, 0xcd, 0xfe, 0x7a, 0x01, 0x09, 0x42, 0xe3, 0xc5, 0xc6, 0x2d, 0xc1, 0xef, 0x7e, 0x90,
			0x5a, 0x2e, 0x71, 0x92, 0x21, 0xf8, 0xb8, 0x3d, 0xe6, 0x57, 0x40, 0x5a, 0xd2, 0x91, 0x65, 0xdf},
		{0x2e, 0x57, 0x50, 0xf4, 0xf3, 0xd2, 0x9d, 0x6f, 0xa7, 0x05, 0xbc, 0x0b, 0x0b, 0x89, 0xea, 0xa6,
			0xd6, 0x8c, 0x20, 0x21, 0x8c, 0x8e, 0x7f, 0x19, 0x03, 0xca, 0xa3, 0x4c, 0xdf, 0xd4, 0x77, 0x7d},
		{0xc5, 0xc4, 0xc0, 0xe2, 0x0c, 0xf3, 0x3c, 0xca, 0xbf, 0x74, 0x12, 0x6a, 0x05, 0x04, 0xc0, 0x62,
			0xed, 0xec, 0x23, 0x08, 0xef, 0xfd, 0xda, 0x5d, 0xb8, 0xec, 0x45, 0x9a, 0x4a, 0x88, 0xcf, 0xe7},
		{0x3c, 0x8a, 0xf5, 0x98, 0x0a, 0x30, 0xce, 0xde, 0x0a, 0x4a, 0x78, 0x62, 0x57, 0xa4, 0xb2, 0x49,
			0x5c, 0x36, 0x05, 0xed, 0xca, 0xc5, 0x85, 0x65, 0x83, 0x90, 0x71, 0xe2, 0xc7, 0x18, 0xd3, 0xdf},
		{0x41, 0x56, 0xd9, 0xdc, 0xe7, 0xc7, 0xeb, 0x9b, 0xc1, 0xdf, 0x68, 0x0b, 0x84, 0x72, 0xfd, 0xfc,
			0x35, 0xd9, 0xde, 0xb9, 0x33, 0xa2, 0xb7, 0x10, 0xe8, 0x72, 0x10, 0xdf, 0x62, 0xdd, 0x9d, 0xba},
		{0x3c, 0x27, 0x7a, 0xb9, 0x06, 0x80, 0x2e, 0x93, 0x66, 0xe0, 0x50, 0x9d, 0x0f, 0x70, 0xed, 0xa2,
			0xf1, 0xd1, 0xaa, 0x37, 0x90, 0xc0, 0x10, 0x03, 0xdb, 0xf5, 0xc1, 0x7e, 0x3e, 0x8c, 0x8c, 0x8e},
		{0xc4, 0xce, 0x52, 0x70, 0xca, 0x07, 0x17, 0x01, 0xed, 0xc9, 0x01, 0x0d, 0x24, 0xdc, 0x09, 0x9d,
			0xaf, 0xc1, 0x71, 0x45, 0xb6, 0xb7, 0x0d, 0xc8, 0x3c, 0x46, 0x18, 0xaf, 0xe0, 0xe2, 0xa0, 0x89},
		{0x28, 0x7e, 0xf4, 0x4c, 0x88, 0xc4, 0xce, 0x55, 0xd1, 0x3f, 0x3a, 0xc6, 0x2a, 0xe4, 0xcd, 0x6a,
			0x6b, 0xb3, 0xe3, 0x97, 0x57, 0x3f, 0x83, 0x1e, 0xaa, 0x10, 0x58, 0xd9, 0x03, 0x75, 0xbf, 0x41},
		{0xeb, 0x63, 0xcd, 0xae, 0xea, 0xce, 0x36, 0xd7, 0x19, 0xac, 0xf4, 0x05, 0x08, 0x25, 0x87, 0x90,
			0x04, 0x00, 0x3c, 0xfa, 0x1b, 0xf2, 0x5c, 0x65, 0x2e, 0xc6, 0x43, 0xce, 0xce, 0x1e, 0x2e, 0x08},
		{0x3d, 0x62, 0xaa, 0x1a, 0xef, 0xd4, 0x31, 0x3c, 0x81, 0xbc, 0xef, 0x6b, 0x20, 0x80, 0x07, 0x63,
			0xdf, 0x5f, 0xb1, 0x1d, 0xac, 0x95, 0xbf, 0x72, 0x33, 0xa0, 0xd6, 0x7f, 0x73, 0xad, 0x56, 0xd0},
		{0xcb, 0x66, 0x46, 0xad, 0x84, 0x9f, 0x88, 0x90, 0xbe, 0x64, 0xcb, 0xde, 0x4f, 0xad, 0x68, 0xa2,
			0x27, 0x08, 0x82, 0x1e, 0x60, 0x70, 0x43, 0x2f, 0x5f, 0x42, 0xdd, 0x73, 0x34, 0x46, 0xc0, 0x81},
		{0xa2, 0xbd, 0x7a, 0x24, 0xc6, 0x9d, 0x2b, 0x25, 0x76, 0xd8, 0xfb, 0x42, 0x71, 0xfc, 0x51, 0x70,
			0x5d, 0xbb, 0x16, 0x5d, 0x79, 0xfb, 0x57, 0xff, 0x24, 0x27, 0x75, 0x15, 0xcd, 0x7f, 0x49, 0xd1},
		{0xee, 0x84, 0x99, 0xc4, 0x4e, 0xb5, 0xb7, 0x08, 0xc3, 0xb7, 0x99, 0xc1, 0xbd, 0xb3, 0xb2, 0x1f,
			0xd6, 0xdb, 0x97, 0x2c, 0x08, 0x1c, 0x52, 0xcc, 0x1c, 0x6e, 0x3d, 0xd6, 0x4d, 0xb4, 0x84, 0xe4},
		{0x2c, 0xdc, 0xe3, 0xd6, 0x97, 0xaf, 0x37, 0x16, 0xa9, 0xb3, 0xcd, 0xf0, 0x68, 0xb4, 0x3e, 0x51,
			0x38, 0x46, 0xe1, 0x7c, 0xc9, 0xfd, 0x42, 0x79, 0x29, 0xaa, 0xd7, 0x01, 0x65, 0xf2, 0x1d, 0xda},
		{0x87, 0xb8, 0x1a, 0xf2, 0xb2, 0xb2, 0x2c, 0xbd, 0xf2, 0x68, 0xe2, 0x11, 0xd0, 0x12, 0xd6, 0x04,
			0x89, 0x2d, 0x3c, 0x94, 0x8f, 0xf2, 0x98, 0xd6, 0x1d, 0x6c, 0x94, 0x2e, 0xee, 0x84, 0x7f, 0x86},
		{0xed, 0x7e, 0x64, 0xaf, 0xf8, 0x7f, 0xb6, 0x36, 0x98, 0x43, 0xe9, 0x30, 0x5b, 0xe1, 0x3d, 0x0d,
			0xaf, 0x34, 0x59, 0xda, 0x4e, 0x1d, 0x25, 0x75, 0xaa, 0x8e, 0x08, 0x80, 0x4e, 0xc0, 0x2d, 0x22},
		{0xd2, 0x80, 0x95, 0xd6, 0xc6, 0x20, 0x35, 0xa5, 0x84, 0xc2, 0x9b, 0x0e, 0x36, 0x17, 0xe0, 0xea,
			0xe3, 0x45, 0x4b, 0x86, 0xfa, 0xe6, 0xcf, 0xfb, 0xf8, 0x88, 0x43, 0x1a, 0x4c, 0x9d, 0xd7, 0x6a},
		{0xc9, 0x47, 0x9e, 0x0e, 0xad, 0xe8, 0xd7, 0x88, 0x7f, 0x0d, 0x03, 0x03, 0x44, 0x6c, 0x07, 0x9b,
			0xbe, 0x4e, 0x54, 0xc1, 0x5d, 0x5b, 0xfb, 0xde, 0x84, 0x38, 0x8a, 0x33, 0xe0, 0xc3, 0x54, 0x8e},
		{0xc9, 0x42, 0xff, 0xfc, 0x60, 0x82, 0x68, 0xbb, 0xd8, 0xf7, 0xaf, 0xb2, 0x3d, 0x9a, 0x4f, 0xde,
			0x41, 0x7d, 0xb6, 0xaa, 0x69, 0xd4, 0x75, 0x25, 0x0a, 0xf8, 0x9c, 0x87, 0xeb, 0x95, 0x58, 0x60},
		{0x98, 0x58, 0xf7, 0x10, 0x59, 0xb2, 0xbc, 0xc1, 0x17, 0xdf, 0x33, 0x3d, 0x2b, 0x37, 0xac, 0x19,
			0x4b, 0xee, 0x70, 0x73, 0x7d, 0x07, 0x58, 0x97, 0x30, 0x50, 0x32, 0xb3, 0x12, 0x85, 0x93, 0x13},
		{0x0e, 0x03, 0xd5, 0xe8, 0xe9, 0x8b, 0x69, 0x15, 0x01, 0xb2, 0x5b, 0xd6, 0x2a, 0xdc, 0x7c, 0x47,
			0x83, 0xd8, 0x14, 0xd1, 0x29, 0x70, 0xc3, 0xe3, 0xff, 0x8a, 0x8b, 0xeb, 0x11, 0x19, 0x59, 0xc2},
		{0x52, 0x41, 0xdc, 0x10, 0xcb, 0x3c, 0x70, 0x0e, 0x46, 0x44, 0x69, 0x43, 0xd2, 0x7b, 0x97, 0x1f,
			0xef, 0xa7, 0xe8, 0x81, 0x15, 0xf8, 0x66, 0xd6, 0xf8, 0x3d, 0x50, 0x2f, 0xf1, 0xbc, 0x06, 0xc2},
	}

	for i := range kat {
		in := make([]byte, i)
		do_hash(t, i, in[:], kat[i])
	}
}

func TestReset(t *testing.T) {

	var d digest
	var out [32]byte
	in := []byte("abc")
	exp := [32]byte{
		0x66, 0xC7, 0xF0, 0xF4, 0x62, 0xEE, 0xED, 0xD9, 0xD1, 0xF2, 0xD4, 0x6B, 0xDC, 0x10, 0xE4, 0xE2,
		0x41, 0x67, 0xC4, 0x87, 0x5C, 0xF2, 0xF7, 0xA2, 0x29, 0x7D, 0xA0, 0x2B, 0x8F, 0x4B, 0xA8, 0xE0,
	}

	d.Init()
	d.Write(in[:1])
	d.Reset()
	d.Write(in[:])
	copy(out[:], d.Sum(nil))
	if out != exp {
		t.Error("Wrong result")
	}
}

func TestSplit(t *testing.T) {

	var d digest
	var out [32]byte
	in := []byte{
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	}

	exp := [32]byte{
		0xde, 0xbe, 0x9f, 0xf9, 0x22, 0x75, 0xb8, 0xa1, 0x38, 0x60, 0x48, 0x89, 0xc1, 0x8e, 0x5a, 0x4d,
		0x6f, 0xdb, 0x70, 0xe5, 0x38, 0x7e, 0x57, 0x65, 0x29, 0x3d, 0xcb, 0xa3, 0x9c, 0x0c, 0x57, 0x32,
	}

	d.Init()
	d.Write(in[:8])
	d.Write(in[8:16])
	copy(out[:], d.Sum(in[16:]))

	if out != exp {
		t.Error("Wrong result")
	}
}

func TestCallSumMultipleTimes(t *testing.T) {
	var out [32]byte

	in := []byte{
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
		0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64, 0x61, 0x62, 0x63, 0x64,
	}

	exp := [32]byte{
		0xde, 0xbe, 0x9f, 0xf9, 0x22, 0x75, 0xb8, 0xa1, 0x38, 0x60, 0x48, 0x89, 0xc1, 0x8e, 0x5a, 0x4d,
		0x6f, 0xdb, 0x70, 0xe5, 0x38, 0x7e, 0x57, 0x65, 0x29, 0x3d, 0xcb, 0xa3, 0x9c, 0x0c, 0x57, 0x32,
	}

	d := New()
	d.Write(in[:10])

	d.Sum(nil) // That's done on purpose
	copy(out[:], d.Sum(in[10:]))

	if out != exp {
		t.Error("Wrong result")
	}
}

/* ------------------ Benchmarks ------------------- */
var bench = New()
var buf = make([]byte, 8192)

func benchmarkSize(b *testing.B, size int) {
	b.SetBytes(int64(size))

	for i := 0; i < b.N; i++ {
		bench.Reset()
		bench.Write(buf[:size])
		bench.Sum(nil)
	}
}

func BenchmarkHash8Bytes(b *testing.B) {
	benchmarkSize(b, 8)
}

func BenchmarkHash1K(b *testing.B) {
	benchmarkSize(b, 1024)
}

func BenchmarkHash8K(b *testing.B) {
	benchmarkSize(b, 8192)
}
