/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fts.h>

#include <QtGui>
//#include <qapplication.h>
//#include <qmessagebox.h>
//#include <qfileinfo.h>
//#include <qdir.h>
//#include <qdeepcopy.h>

#include "gui_user.h"
#include "backend/iointerface.h"
#include "backend/ifpinterface.h"
#include "ifpguiview.h"

QList<FileInfos> new_ifp_file_tree;
QString iFPNotConnectedMsg = QObject::tr("Could not communicate with iRiver device.\n"
                                         "Try unplugging the USB link, powering off the device, and plugging it back in.\n\n"
                                         "After you have reconnected, double-click the root ('\\') directory in the iRiver Filesystem panel.");


//-----------------------------------------------------------
// helper functions
//-----------------------------------------------------------
int get_next_filedir_basename(char *str);

void slash2backslash(QString& str);
int ifp_get_size_recursive(QList<FileInfos> *ifp_fi, int *filecnt);



//===========================================================================
// file_dir_callbk()
//   This is the ifp list callback.  This is used to build the file tree
//   which is stored in ifp_file_tree;
//
//   context  : the current node to populate in the file tree.
//   type     : the type of node (file or directory)
//   filename : the name of the node
//   filesize : the size of the file if the node is a file
//
//   Returns: 0 on success
//===========================================================================
static int file_dir_callbk(void * context, int type, const char * filename, int filesize)
{
    QList<FileInfos> *list;
    FileInfos fi;

    list = (QList<FileInfos> *)context;

    if (list == NULL)
        return(-1);

    fi = FileInfos(QString::fromUtf8(filename, -1), type, filesize, NULL);
    list->append(fi);
    return 0;
}



//===========================================================================
// get_next_filedir_basename()
//   This function returns the first position of a dir seperator char.
//===========================================================================
int get_next_filedir_basename(QString str)
{
    int ret;

    ret = str.indexOf("\\", 0);
    if (ret == -1)
        ret = str.indexOf("/", 0);
    return(ret);
}

//===========================================================================
// ifp_get_size_recursive()
//   This function gets size of the directory @ifp_fi recursively.
//
//   Returns: size on success
//===========================================================================
int ifp_get_size_recursive(FileInfos *finfo, int *filecnt)
{
    int size;
    QList<FileInfos> *list;
    QList<FileInfos>::iterator fi_iter;

    size = 0;
    list = finfo->sub_file_tree;
    for (fi_iter= list->begin(); fi_iter != list->end(); ++fi_iter)
    {
        if (fi_iter->FileType() == IFP_FILE)
        {
            size += fi_iter->FileSize();
            (*filecnt) += 1;
        }
        else if (fi_iter->FileType() == IFP_DIR)
        {
            size += ifp_get_size_recursive(&(*fi_iter), filecnt);
        }
    }
    return(size);
}


//===========================================================================
// local_get_dir_size()
//   This function gets size of the directory @filename.  All its files
//   and subdirectories with their files.
//
//   Returns: filesize on success
//            -ENOENT for filename not found
//===========================================================================
int local_get_dir_size(QString dirName, int *filecnt)
{
    int filesize, cnt, tmpcnt;

    QDir dir(dirName, "", QDir::Name | QDir::IgnoreCase, QDir::AllEntries);
    if (!dir.exists())
        return(0);

    QFileInfoList list = dir.entryInfoList();
    QFileInfo finfo;

    filesize = 0;
    for (cnt = 0; cnt < list.size(); cnt++)
    {
        finfo = list.at(cnt);
        if (finfo.isFile())
        {
            filesize += finfo.size();
            *filecnt += 1;
        }
        else
        {
            if ((finfo.fileName() != ".") && (finfo.fileName() != ".."))
            {
                tmpcnt = 0;
                filesize += local_get_dir_size(finfo.filePath(), &tmpcnt);
                *filecnt += tmpcnt;
            }
        }
    }
    return(filesize);
}



int showYesNoAllMsgBox(const QString& msgH, const QString& msgB)
{
    int mbret;

    QMessageBox mb(msgH, msgB,
                   QMessageBox::Question,
                   QMessageBox::No  | QMessageBox::Escape,
                   QMessageBox::Yes | QMessageBox::Default,
                   QMessageBox::YesAll);
    mbret = mb.exec();
    return(mbret);
}

int showYesNoMsgBox(const QString& msgH, const QString& msgB)
{
    int mbret;

    QMessageBox mb(msgH, msgB,
                   QMessageBox::Question,
                   QMessageBox::No  | QMessageBox::Escape,
                   QMessageBox::Yes | QMessageBox::Default,
                   0);
    mbret = mb.exec();
    return(mbret);
}

void showInfoMsgBox(const QString& msgH, const QString& msgB)
{
    QMessageBox mb(msgH, msgB,
                   QMessageBox::Information,
                   QMessageBox::Ok, 0, 0);
    mb.exec();
    return;
}

void backslash2slash(QString& str)
{
    str.replace('\\', '/');
}

void slash2backslash(QString& str)
{
    str.replace('/', '\\');
}


//===========================================================================
// ifp_get_info()
//   This function gets the FileInfos for the given @filename.
//
//   Returns: IFP_OK on success
//            -ENOENT for filename not found
//===========================================================================
int TiFPgetInfo::iFPgetInfo(QString filename, FileInfos& finfo)
{
    t_filename = filename;
    t_finfo = finfo;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    finfo = t_finfo;
    return (t_result);
}

void TiFPgetInfo::run()
 {
    int cnt;
    QString tmp_name, scratch_name, split_str;
    FileInfos fi_temp;
    QList<FileInfos> *list;
    QList<FileInfos>::iterator fi_iter;

    t_result = -ENOENT;

    if ((t_filename == "\\") || (t_filename == "/"))
    {
        t_finfo = new_ifp_file_tree.first();
        t_result = IFP_OK;
        return;
    }
    if (new_ifp_file_tree.isEmpty())
        return;

    // create the path list
    split_str = t_filename.left(1);
    QStringList lst(t_filename.split(split_str));
    // start with the node after the root node (\)
    fi_temp = new_ifp_file_tree.first();

    QStringList::Iterator it2 = lst.begin();
    for ( ; it2 != lst.end(); ++it2)
    {
        list = fi_temp.sub_file_tree;
        if (list == NULL)
            return;
        if (list->isEmpty())
            return;

        // get the name of the node to find
        tmp_name = *it2;
        // find the node in the list
        for (cnt=0; cnt < list->count(); cnt++)
        {
            if (list->at(cnt).FileName() == tmp_name)
            {
                t_finfo = list->at(cnt);
                t_result = IFP_OK;
                break;
            }
        }
        if (t_result == IFP_OK)
            break;
    }
    return;
}


//===========================================================================
// iFPreadFileTree()
//   This function retrieves all files on the iFP device.  It stores them
//   in the ifp_file_tree structure.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPreadFileTree::iFPreadFileTree()
{
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    return (t_result);
}

void TiFPreadFileTree::run()
{
    QString start_dir;
    FileInfos fi_temp;

    new_ifp_file_tree.clear();

    // the list owns the objects
    //new_ifp_file_tree.setAutoDelete(TRUE);

    // Add the first element "/"
    fi_temp = FileInfos("/", IO_INTF_DIR, 0, NULL);
    fi_temp.sub_file_tree = new QList<FileInfos>;
    start_dir = "/";
    t_result = iFPgetDirList(&ifp_dev, start_dir, fi_temp.sub_file_tree);
    new_ifp_file_tree.append(fi_temp);

    return;
}

//===========================================================================
// iFPgetDirList()
//   This function retrieves all files in the directory specified by
//   @filename and fills in the @file_info_ptr parameter with the infos
//   about every entry.  Also, it goes through the list and gets the info
//   (files and directories) for each subdirectiry.
//
//   dev            : the ifp_device structure
//   filename       : the name of the directory to get the list of
//   file_finfo_ptr : the address of the structure to place the info into
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPreadFileTree::iFPgetDirList(struct ifp_device * dev, QString& filename, QList<FileInfos> *list)
{
    int ret;
    QString fullname, tmp_str;
    //char fullname[IFP_MAXPATHLEN];
    QList<FileInfos>::iterator fi_iter;

    slash2backslash(filename);

    // check if directory exists
    if (ifp_exists(dev, filename.toUtf8().data()) != IFP_DIR)
    {
        return (-ENOENT);
    }

    ret = ifp_list_dirs(dev, filename.toUtf8().data(), file_dir_callbk, list);
    if (ret)
    {
        ifp_err_i(ret, "couldn't get basic directory listing.");
        return IFP_ERR_DEV_FUBAR;
    }

    // check if directory is empty
    if (list->isEmpty())
    {
        return(IFP_OK);
    }

    // now look through results and get the subdirectories
    for (fi_iter= list->begin(); fi_iter != list->end(); ++fi_iter)
    {
        if (fi_iter->FileType() == IO_INTF_DIR)
        {
            fullname = filename;
            if (!fullname.endsWith("\\"))
                fullname.append("\\");
            fullname.append(fi_iter->FileName());
            fullname.truncate(IFP_MAXPATHLEN);
            // get the subdirectiry info
            fi_iter->sub_file_tree = new QList<FileInfos>;
            ret = iFPgetDirList(dev, fullname, fi_iter->sub_file_tree);
            if (ret != IFP_OK)
                return(ret);
        }
    }
    return(IFP_OK);
}

//===========================================================================
// iFPgetDirSize()
//   This function gets size of the directory @filename.  All its files
//   and subdirectories with their files.
//
//   Returns: filesize on success
//            -ENOENT for filename not found
//===========================================================================
int TiFPgetDirSize::iFPgetDirSize(QString filename, int *filecnt)
{
    t_filename = filename;
    t_filecnt = *filecnt;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    *filecnt = t_filecnt;
    return (t_result);
}

void TiFPgetDirSize::run()
{
    FileInfos fi_temp;
    TiFPgetInfo t_info; // = new TiFPgetInfo();

    t_result = t_info.iFPgetInfo(t_filename, fi_temp);

    if (t_result != IFP_OK)
        return;

    if (fi_temp.FileType() == IFP_FILE)
    {
        t_result = -ENOENT;
        return;
    }

    // get the size
    t_result = ifp_get_size_recursive(&fi_temp, &t_filecnt);
    return;
}

//===========================================================================
// iFPgetFileList()
//   This function gets the list of files and sub directories for a given
//   file.  The data is gathered from the local copy and not queried from
//   the iFP device.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPgetFileList::iFPgetFileList(const QString& filename, QList<FileInfos> *list)
{
    t_filename = filename;
    t_list = list;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    return (t_result);
}

void TiFPgetFileList::run()
{
    FileInfos finfo, nfinfo;
    QList<FileInfos> *tmp_list;
    QList<FileInfos>::iterator fi_iter;
    TiFPgetInfo t_info; // = new TiFPgetInfo();

    t_result = t_info.iFPgetInfo(t_filename, finfo);
    if (t_result != IFP_OK)
        return;

    tmp_list = finfo.sub_file_tree;

    if (tmp_list == NULL)
    {
        t_result = IFP_OK;
        return;
    }

    for (fi_iter= tmp_list->begin(); fi_iter != tmp_list->end(); ++fi_iter)
    {
        nfinfo = FileInfos(fi_iter->FileName(),
                           fi_iter->FileType(),
                           fi_iter->FileSize(),
                           NULL);
        t_list->append(nfinfo);
    }
    return;
}

//===========================================================================
// iFPdownloadFile()
//   Download the file sfilename from the iFP device to the local directory destDir
//
// @param sfilename the source filename on the iFP device.  Must be / delimited.
// @param destDir the local directory; the place to save the downloaded file
// @param fn_context the current context of the transfer.  User defined structure
// usually used to keep track of the progress of the download.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPdownloadFile::iFPdownloadFile(const QString& sfilename, const QString& destDir, void* fn_context)
{
    t_sfilename = sfilename;
    t_destDir = destDir;
    t_fn_context = fn_context;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    return (t_result);

}

void TiFPdownloadFile::run()
{
    int pos;
    QByteArray sbuffer, dbuffer;
    QString Message, destination, baseName, err, ifpSourceName;

    t_result = -1;
    if (ifp_dh == NULL)
        return;

    baseName = t_sfilename;
    pos = baseName.lastIndexOf('/');
    baseName.remove(0, pos + 1);
    destination = t_destDir + baseName;

    ifpSourceName = t_sfilename;
    slash2backslash(ifpSourceName);

    sbuffer = ifpSourceName.toUtf8();
    dbuffer = destination.toUtf8();
    t_result = ifpgui_ifp_download_file(&ifp_dev, sbuffer, dbuffer, progress, t_fn_context);

    return;
}

//===========================================================================
// iFPdownloadDirectory()
//   Download the directory sdirname from the iFP device to the local directory destDir
//
// @param sdirname the source directory name on the iFP device.  Must be / delimited.
// @param destDir the local directory; the place to save the downloaded directory
// @param fn_context the current context of the transfer.  User defined structure
// usually used to keep track of the progress of the download.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPdownloadDirectory::iFPdownloadDirectory(const QString& sdirname, const QString& destDir, void* fn_context)
{
    t_sdirname = sdirname;
    t_destDir = destDir;
    t_fn_context = fn_context;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    return (t_result);
}

void TiFPdownloadDirectory::run()
{
    int pos;
    QByteArray sbuffer, dbuffer;
    QString Message, destination, baseName, err, ifpSourceName;

    t_result = -1;
    if (ifp_dh == NULL)
        return;

    baseName = t_sdirname;
    pos = baseName.lastIndexOf('/');
    baseName.remove(0, pos + 1);
    destination = t_destDir + baseName;

    ifpSourceName = t_sdirname;
    slash2backslash(ifpSourceName);

    sbuffer = ifpSourceName.toUtf8();
    dbuffer = destination.toUtf8();
    t_result = ifpgui_ifp_download_dir(&ifp_dev, sbuffer, dbuffer, progress, t_fn_context);

    return;
}


//===========================================================================
// iFPuploadFile()
//   Upload the file sfilename from the iFP device to the iRiver directory destDir
//
// @param sdirname the source directory name on the iFP device.  Must be / delimited.
// @param destDir the local directory; the place to save the downloaded directory
// @param fn_context the current context of the transfer.  User defined structure
// usually used to keep track of the progress of the download.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPuploadFile::iFPuploadFile(const QString& sfilename, const QString& destDir, void* fn_context)
{
    t_sfilename = sfilename;
    t_destDir = destDir;
    t_fn_context = fn_context;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    return (t_result);
}

void TiFPuploadFile::run()
{
    int pos;
    QByteArray sbuffer, dbuffer;
    QString Message, destination, baseName, err, ifpDestName;

    t_result = -1;
    if (ifp_dh == NULL)
        return;

    baseName = t_sfilename;
    pos = baseName.lastIndexOf('/');
    baseName.remove(0, pos + 1);
    destination = t_destDir + baseName;

    ifpDestName = destination;
    slash2backslash(ifpDestName);

    sbuffer = t_sfilename.toUtf8();
    dbuffer = ifpDestName.toUtf8();
    t_result = ifpgui_ifp_upload_file(&ifp_dev, sbuffer, dbuffer, progress, t_fn_context);

    return;
}

//===========================================================================
// iFPuploadFile()
//   Upload the directory sdirname to the iRiver device directory destDir
//
// @param sdirname the source directory name on the iFP device.  Must be / delimited.
// @param destDir the local directory; the place to save the downloaded directory
// @param fn_context the current context of the transfer.  User defined structure
// usually used to keep track of the progress of the download.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int TiFPuploadDirectory::iFPuploadDirectory(const QString& sdirname, const QString& destDir, void* fn_context)
{
    t_sdirname = sdirname;
    t_destDir = destDir;
    t_fn_context = fn_context;
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(10);
    }
    return (t_result);
}

void TiFPuploadDirectory::run()
{
    int pos;
    QByteArray sbuffer, dbuffer;
    QString Message, destination, baseName, err, ifpDestName;

    t_result = -1;
    if (ifp_dh == NULL)
        return;

    baseName = t_sdirname;
    pos = baseName.lastIndexOf('/');
    baseName.remove(0, pos + 1);
    destination = t_destDir + baseName;

    ifpDestName = destination;
    slash2backslash(ifpDestName);

    sbuffer = t_sdirname.toUtf8();
    dbuffer = ifpDestName.toUtf8();
    t_result = ifpgui_ifp_upload_dir(&ifp_dev, sbuffer, dbuffer, progress, t_fn_context);

    return;
}

//===========================================================================
// TiFPformatDevice()
//   Reformats the device's storage media. (Deletes all your stuff.)
//
//   Returns: IFP_OK if successful,
//            1 on  error
//===========================================================================
int TiFPformatDevice::iFPformatDevice()
{
    start();
    while (isRunning())
    {
        qApp->processEvents();
        msleep(1);
    }
    return (t_result);
}

void TiFPformatDevice::run()
{
    t_result = ifpgui_ifp_format(&ifp_dev);
}

