/*
 * Copyright (C) 2013 ~ 2019 National University of Defense Technology(NUDT) & Tianjin Kylin Ltd.
 *
 * Authors:
 *  Kobe Lee    lixiang@kylinos.cn/kobe24_lixiang@126.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "promptwidget.h"

#include <QApplication>
#include <QDebug>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QPainter>

PromptWidget::PromptWidget(QWidget *parent) : QFrame(parent)
    , m_iconLabel(new QLabel(this))
    , m_textLabel(new QLabel(this))
    , m_closeBtn(new QPushButton(this))
{
    this->setStyleSheet("QFrame{background:rgba(0,0,0,0.85);} QFrame:hover{background:rgba(0, 0, 0, 0.6);}");
    this->setFixedSize(134, 99);
    this->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    this->setContentsMargins(0, 0, 0, 0);

    QVBoxLayout *m_vlayout = new QVBoxLayout(this);
    m_vlayout->setContentsMargins(10, 10, 10, 10);
    m_vlayout->setSpacing(5);

    m_closeBtn->setFixedSize(12, 12);
    m_closeBtn->setFocusPolicy(Qt::NoFocus);
    m_closeBtn->setStyleSheet("QPushButton{background-image:url(':/res/delete_normal.png');border:0px;}QPushButton:hover{background:url(':/res/delete_hover_press.png');}QPushButton:pressed{background:url(':/res/delete_hover_press.png');}");
    m_closeBtn->setVisible(false);

    m_iconLabel->setFixedSize(45, 44);
    m_iconLabel->setStyleSheet("QLabel{border:none;background-color:transparent;}");

    m_textLabel->setStyleSheet("QLabel{border:none;background-color:transparent;color:#ffffff;font-size:14px;}");
    m_textLabel->setWordWrap(true);
    m_textLabel->setAlignment(Qt::AlignCenter);
    m_textLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    m_vlayout->addWidget(m_iconLabel, 0, Qt::AlignHCenter);
    m_vlayout->addWidget(m_textLabel, 0, Qt::AlignHCenter);

    connect(m_closeBtn, &QPushButton::clicked, this, [=] {
        this->setVisible(false);
    });
}

PromptWidget::PromptWidget(const QString &text, QWidget *parent, const QString &movieSouce, bool movie)
    : QFrame(parent)
    , m_iconLabel(new QLabel(this))
    , m_textLabel(new QLabel(this))
    , m_closeBtn(new QPushButton(this))
    , m_isMovie(movie)
{
    this->setStyleSheet("QFrame{background:rgba(0,0,0,0.85);} QFrame:hover{background:rgba(0, 0, 0, 0.6);}");
    this->setFixedSize(134, 50);
    this->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    this->setContentsMargins(0, 0, 0, 0);

    QHBoxLayout *m_hlayout = new QHBoxLayout(this);
    m_hlayout->setContentsMargins(5, 10, 5, 10);
    m_hlayout->setSpacing(5);

    m_closeBtn->setFixedSize(12, 12);
    m_closeBtn->setFocusPolicy(Qt::NoFocus);
    m_closeBtn->setStyleSheet("QPushButton{background-image:url(':/res/delete_normal.png');border:0px;}QPushButton:hover{background:url(':/res/delete_hover_press.png');}QPushButton:pressed{background:url(':/res/delete_hover_press.png');}");
    m_closeBtn->setVisible(false);

    m_iconLabel->setFixedSize(16, 16);
    m_iconLabel->setStyleSheet("QLabel{border:none;background-color:transparent;}");

    m_textLabel->setStyleSheet("QLabel{border:none;background-color:transparent;color:#ffffff;font-size:14px;}");
    m_textLabel->setText(text);
    QFont font = m_textLabel->font();
    const QFontMetrics fm(font);
    m_textLabel->setFixedWidth(fm.width(m_textLabel->text()));

    if (m_isMovie) {
        m_movie = new QMovie(movieSouce);
        m_iconLabel->setMovie(m_movie);
        m_movie->start();
    }

    m_hlayout->addStretch();
    m_hlayout->addWidget(m_textLabel, 0, Qt::AlignVCenter | Qt::AlignRight);
    m_hlayout->addWidget(m_iconLabel, 0, Qt::AlignVCenter | Qt::AlignLeft);
    m_hlayout->addStretch();

    connect(m_closeBtn, &QPushButton::clicked, this, [=] {
        m_movie->stop();
        this->setVisible(false);
    });
}

PromptWidget::~PromptWidget()
{
    if (m_movie) {
        delete m_movie;
    }
}

void PromptWidget::setIconAndText(const QString &iconPath, const QString &text)
{
    const qreal ratio = qApp->devicePixelRatio();
    QIcon icon = QIcon(iconPath);
    QPixmap pixmap = icon.pixmap(m_iconLabel->width()*ratio, m_iconLabel->height()*ratio);
    pixmap = pixmap.scaled(m_iconLabel->width()*ratio, m_iconLabel->height()*ratio, Qt::KeepAspectRatioByExpanding, Qt::SmoothTransformation);
    pixmap.setDevicePixelRatio(ratio);
    m_iconLabel->setPixmap(pixmap);

    m_textLabel->setText(text);
}

void PromptWidget::enterEvent(QEvent *event)
{
    QFrame::enterEvent(event);

    m_closeBtn->move(this->width() -12, 0);
    m_closeBtn->setVisible(true);
}

void PromptWidget::leaveEvent(QEvent *event)
{
    QFrame::enterEvent(event);

    m_closeBtn->setVisible(false);
}

bool PromptWidget::event(QEvent *event)
{
    switch (event->type()) {
    case QEvent::MouseButtonPress:
        if (m_isMovie && m_movie) {
            //m_movie->stop();
        }
        else {
            this->setVisible(false);
        }
        break;
    /*case QEvent::Show:
        if (m_isMovie && m_movie) {
            m_movie->start();
            this->setVisible(true);
        }
        break;*/
    default:
        break;
    }

    return QFrame::event(event);
}


void PromptWidget::showEvent(QShowEvent *event)
{
    QFrame::showEvent(event);

    if (m_isMovie && m_movie) {
        m_movie->start();
    }
}

void PromptWidget::hideEvent(QHideEvent* event)
{
  QFrame::hideEvent(event);

  if (m_isMovie && m_movie) {
      m_movie->stop();
  }
}
