// -*- c++ -*-
/* This file is part of the KDE libraries
    Copyright (C) 1997, 1998 Richard Moore <rich@kde.org>
                  1998 Stephan Kulow <coolo@kde.org>
                  1998 Daniel Grana <grana@ie.iwi.unibe.ch>
                  2000,2001 Carsten Pfeiffer <pfeiffer@kde.org>
                  2001 Frerich Raabe <raabe@kde.org>
                  2007 David Faure <faure@kde.org>
                  2009 David Jarvie <djarvie@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KFILEDIALOG_H
#define KFILEDIALOG_H

#include <kdelibs4support_export.h>
#include <QDialog>
#include <kfile.h>
#include <qmimetype.h>
#include <QUrl>

class KFileWidget;

class KActionCollection;
class KUrlComboBox;
class KFileFilterCombo;
class QPushButton;
class KToolBar;
class KPreviewWidgetBase;

class KFileDialogPrivate;

/**
 * @warning This class should be avoided in new code.
 * QFileDialog should be used instead.
 * @see QFileDialog
 *
 * Provides a user (and developer) friendly way to
 * select files and directories.
 *
 * The widget can be used as a drop in replacement for the
 * QFileDialog widget, but has greater functionality and a nicer GUI.
 *
 * You will usually want to use one of the static methods
 * getOpenFileName(), getSaveFileName(), getOpenUrl()
 * or for multiple files getOpenFileNames() or getOpenUrls().
 *
 * The dialog has been designed to allow applications to customize it
 * by subclassing. It uses geometry management to ensure that subclasses
 * can easily add children that will be incorporated into the layout.
 *
 * \image html kfiledialog.png "KDE File Dialog"
 *
 * @short A file selection dialog.
 *
 * @deprecated since 5.0, use the QFileDialog API instead.
 * Note that when the KDE QPA theme plugin is installed, the KFileWidget will then
 * be used automatically.
 */
class KDELIBS4SUPPORT_DEPRECATED_EXPORT KFileDialog : public QDialog
{
    Q_OBJECT

public:

    /**
     * Defines some default behavior of the filedialog.
     * E.g. in mode @p Opening and @p Saving, the selected files/urls will
     * be added to the "recent documents" list. The Saving mode also implies
     * setKeepLocation() being set.
     *
     * @p Other means that no default actions are performed.
     *
     * @see setOperationMode
     * @see operationMode
     */
    enum OperationMode { Other = 0, Opening, Saving };

    /**
     * Defines the options to use when calling getSave* functions.
     * @since 4.4
     */
    enum Option {
        ConfirmOverwrite  = 0x01,   /**< Confirm whether to overwrite file to save. */
        ShowInlinePreview = 0x02    /**< Always show an inline preview. */
    };
    Q_DECLARE_FLAGS(Options, Option)

    /**
      * Constructs a file dialog.
      *
      * @param startDir Specifies the starting directory and/or initially selected
      *                 file name, or a last used directory and optional file name
      *                 using the @c kfiledialog:/// syntax.
      *                 Refer to the KFileWidget documentation for more information
      *                 on this parameter.
      *
      * @param filter A shell glob or a mimetype filter that specifies
      *               which files to display. For better consistency across applications,
      *               it is recommended to use a mimetype filter.
      * See setFilter() and setMimeFilter() for details on how to use this argument.
      *
      * @param parent The parent widget of this dialog
      *
      * @param widget A widget, or a widget of widgets, for displaying custom
      *               data in the dialog. This can be used, for example, to
      *               display a check box with the caption "Open as read-only".
      *               When creating this widget, you don't need to specify a parent,
      *               since the widget's parent will be set automatically by KFileDialog.
      *
      * @see KFileWidget::KFileWidget()
      */
    KFileDialog(const QUrl &startDir, const QString &filter,
                QWidget *parent, QWidget *widget = nullptr);

    /**
     * Destructs the file dialog.
     */
    ~KFileDialog();

    /**
     * @returns The selected fully qualified filename.
     */
    QUrl selectedUrl() const;

    /**
     * @returns The list of selected URLs.
     */
    QList<QUrl> selectedUrls() const;

    /**
     * @returns the currently shown directory.
     */
    QUrl baseUrl() const;

    /**
     * Returns the full path of the selected file in the local filesystem.
     * (Local files only)
     */
    QString selectedFile() const;

    /**
     * Returns a list of all selected local files.
     */
    QStringList selectedFiles() const;

    /**
     * Sets the directory to view.
     *
     * @param url URL to show.
     * @param clearforward Indicates whether the forward queue
     * should be cleared.
     */
    void setUrl(const QUrl &url, bool clearforward = true);

    /**
     * Sets the file name to preselect to @p name
     *
     * This takes absolute URLs and relative file names.
     */
    void setSelection(const QString &name);

    /**
     * Sets the operational mode of the filedialog to @p Saving, @p Opening
     * or @p Other. This will set some flags that are specific to loading
     * or saving files. E.g. setKeepLocation() makes mostly sense for
     * a save-as dialog. So setOperationMode( KFileDialog::Saving ); sets
     * setKeepLocation for example.
     *
     * The mode @p Saving, together with a default filter set via
     * setMimeFilter() will make the filter combobox read-only.
     *
     * The default mode is @p Opening.
     *
     * Call this method right after instantiating KFileDialog.
     *
     * @see operationMode
     * @see KFileDialog::OperationMode
     */
    void setOperationMode(KFileDialog::OperationMode);

    /**
     * @returns the current operation mode, Opening, Saving or Other. Default
     * is Other.
     *
     * @see operationMode
     * @see KFileDialog::OperationMode
     */
    OperationMode operationMode() const;

    /**
     * Sets whether the filename/url should be kept when changing directories.
     * This is for example useful when having a predefined filename where
     * the full path for that file is searched.
     *
     * This is implicitly set when operationMode() is KFileDialog::Saving
     *
     * getSaveFileName() and getSaveUrl() set this to true by default, so that
     * you can type in the filename and change the directory without having
     * to type the name again.
     */
    void setKeepLocation(bool keep);

    /**
     * @returns whether the contents of the location edit are kept when
     * changing directories.
     */
    bool keepsLocation() const;

    /**
     * Sets the filter to be used to @p filter.
     *
     * The filter can be either set as a space-separated list of
     * mimetypes, which is recommended, or as a list of shell globs
     * separated by @c '\\n'.
     *
     * If the filter contains an unescaped @c '/', a mimetype filter is assumed.
     * If you would like a @c '/' visible in your filter it can be escaped with
     * a @c '\'. You can specify multiple mimetypes like this (separated with
     * space):
     *
     * \code
     * kfile->setFilter( "image/png text/html text/plain" );
     * \endcode
     *
     * When showing the filter to the user, the mimetypes will be automatically
     * translated into their description like `PNG image'. Multiple mimetypes
     * will be automatically summarized to a filter item `All supported files'.
     * To add a filter item for all files matching @c '*', add @c application/octet-stream
     * as mimetype.
     *
     * If the filter contains no unescaped @c '/', it is assumed that
     * the filter contains conventional shell globs. Several filter items
     * to select from can be separated by @c '\\n'. Every
     * filter entry is defined through @c namefilter|text to display.
     * If no @c '|' is found in the expression, just the namefilter is
     * shown. Examples:
     *
     * \code
     * kfile->setFilter("*.cpp|C++ Source Files\n*.h|Header files");
     * kfile->setFilter("*.cpp");
     * kfile->setFilter("*.cpp|Sources (*.cpp)");
     * kfile->setFilter("*.cpp|" + i18n("Sources (*.cpp)"));
     * kfile->setFilter("*.cpp *.cc *.C|C++ Source Files\n*.h *.H|Header files");
     * \endcode
     *
     * Note: The text to display is not parsed in any way. So, if you
     * want to show the suffix to select by a specific filter, you must
     * repeat it.
     *
     * For better consistency across applications, it is recommended to use a
     * mimetype filter.
     *
     * @see filterChanged
     * @see setMimeFilter
     */
    void setFilter(const QString &filter);

    /**
     * Returns the current filter as entered by the user or one of the
     * predefined set via setFilter().
     *
     * @see setFilter()
     * @see filterChanged()
     */
    QString currentFilter() const;

    /**
     * Returns the mimetype for the desired output format.
     *
     * This is only valid if setMimeFilter() has been called
     * previously.
     *
     * @see setFilterMimeType()
     */
    QMimeType currentFilterMimeType();

    /**
     * Sets the filter up to specify the output type.
     *
     * @param types a list of mimetypes that can be used as output format
     * @param defaultType the default mimetype to use as output format, if any.
     * If @p defaultType is set, it will be set as the current item.
     * Otherwise, a first item showing all the mimetypes will be created.
     * Typically, @p defaultType should be empty for loading and set for saving.
     *
     * Do not use in conjunction with setFilter()
     */
    void setMimeFilter(const QStringList &types,
                       const QString &defaultType = QString());

    /**
     * The mimetype for the desired output format.
     *
     * This is only valid if setMimeFilter() has been called
     * previously.
     *
     * @see setMimeFilter()
     */
    QString currentMimeFilter() const;

    /**
     *  Clears any mime- or namefilter. Does not reload the directory.
     */
    void clearFilter();

    /**
     * Adds a preview widget and enters the preview mode.
     *
     * In this mode the dialog is split and the right part contains your
     * preview widget.
     *
     * Ownership is transferred to KFileDialog. You need to create the
     * preview-widget with "new", i.e. on the heap.
     *
     * @param w The widget to be used for the preview.
     */
    void setPreviewWidget(KPreviewWidgetBase *w);

    /**
     * Forces the inline previews to be shown or hidden, depending on @p show.
     *
     * @param show Whether to show inline previews or not.
     * @since 4.2
     */
    void setInlinePreviewShown(bool show);

    /**
     * Sets whether the dialog should ask before accepting the selected file
     * when KFileDialog::OperationMode is set to Saving.
     *
     * In this case a KMessageBox appears for confirmation.
     *
     * @param enable Set this to true to enable checking.
     * @since 4.2
     */
    void setConfirmOverwrite(bool enable);

    /** @see QWidget::sizeHint() */
    QSize sizeHint() const override;

    /**
     * Creates a modal file dialog and return the selected
     * filename or an empty string if none was chosen.
     *
     * Note that with
     * this method the user must select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getOpenFileName(parent, caption, startDir, filter)
     */
    static QString getOpenFileName(const QUrl &startDir = QUrl(),
                                   const QString &filter = QString(),
                                   QWidget *parent = nullptr,
                                   const QString &caption = QString());

    /**
      * Use this version only if you have no QWidget available as
      * parent widget. This can be the case if the parent widget is
      * a widget in another process or if the parent widget is a
      * non-Qt widget. For example, in a GTK program.
     */
    static QString getOpenFileNameWId(const QUrl &startDir,
                                      const QString &filter,
                                      WId parent_id, const QString &caption);

    /**
     * Creates a modal file dialog and returns the selected
     * filenames or an empty list if none was chosen.
     *
     * Note that with
     * this method the user must select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getOpenFileNames(parent, caption, startDir, filter)
     */
    static QStringList getOpenFileNames(const QUrl &startDir = QUrl(),
                                        const QString &filter = QString(),
                                        QWidget *parent = nullptr,
                                        const QString &caption = QString());

    /**
     * Creates a modal file dialog and returns the selected
     * URL or an empty string if none was chosen.
     *
     * Note that with
     * this method the user must select an existing URL.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getOpenFileUrl(parent, caption, startDir, filter)
     */
    static QUrl getOpenUrl(const QUrl &startDir = QUrl(),
                           const QString &filter = QString(),
                           QWidget *parent = nullptr,
                           const QString &caption = QString());

    /**
     * Creates a modal file dialog and returns the selected
     * URLs or an empty list if none was chosen.
     *
     * Note that with
     * this method the user must select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getOpenFileUrls(parent, caption, startDir, filter)
     */
    static QList<QUrl> getOpenUrls(const QUrl &startDir = QUrl(),
                                   const QString &filter = QString(),
                                   QWidget *parent = nullptr,
                                   const QString &caption = QString());

    /**
     * Creates a modal file dialog and returns the selected
     * filename or an empty string if none was chosen.
     *
     * Note that with this
     * method the user need not select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getSaveFileName(parent, caption, startDir, filter)
     */
    static QString getSaveFileName(const QUrl &startDir = QUrl(),
                                   const QString &filter = QString(),
                                   QWidget *parent = nullptr,
                                   const QString &caption = QString());

    /**
     * Creates a modal file dialog and returns the selected
     * filename or an empty string if none was chosen.
     *
     * Note that with this
     * method the user need not select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     * @param options Dialog options.
     *
     * @see KFileWidget::KFileWidget()
     *
     * @since 4.4
     * @deprecated use QFileDialog::getSaveFileName(parent, caption, startDir, filter, [selectedFilter], options)
     */
    static QString getSaveFileName(const QUrl &startDir,
                                   const QString &filter,
                                   QWidget *parent,
                                   const QString &caption,
                                   Options options);

    /**
     * This function accepts the window id of the parent window, instead
     * of QWidget*. It should be used only when necessary.
     */
    static QString getSaveFileNameWId(const QUrl &startDir, const QString &filter,
                                      WId parent_id,
                                      const QString &caption);

    /**
     * This function accepts the window id of the parent window, instead
     * of QWidget*. It should be used only when necessary.
     *
     * @since 4.4
     */
    static QString getSaveFileNameWId(const QUrl &startDir, const QString &filter,
                                      WId parent_id,
                                      const QString &caption,
                                      Options options);

    /**
     * Creates a modal file dialog and returns the selected
     * filename or an empty string if none was chosen.
     *
     * Note that with this
     * method the user need not select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getSaveFileUrl(parent, caption, startDir, filter)
     */
    static QUrl getSaveUrl(const QUrl &startDir = QUrl(),
                           const QString &filter = QString(),
                           QWidget *parent = nullptr,
                           const QString &caption = QString());

    /**
     * Creates a modal file dialog and returns the selected
     * filename or an empty string if none was chosen.
     *
     * Note that with this
     * method the user need not select an existing filename.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param filter A shell glob or a mimetype filter that specifies which files to display.
     *    The preferred option is to set a list of mimetype names, see setMimeFilter() for details.
     *    Otherwise you can set the text to be displayed for the each glob, and
     *    provide multiple globs, see setFilter() for details.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     * @param options Dialog options.
     *
     * @see KFileWidget::KFileWidget()
     *
     * @since 4.4
     * @deprecated use QFileDialog::getSaveFileUrl(parent, caption, startDir, filter, [selectedFilter], options)
     */
    static QUrl getSaveUrl(const QUrl &startDir,
                           const QString &filter,
                           QWidget *parent,
                           const QString &caption,
                           Options options);

    /**
     * Creates a modal directory-selection dialog and returns the selected
     * directory (local only) or an empty string if none was chosen.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     * @return the path to an existing local directory.
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getExistingDirectory(parent, caption, startDir)
     */
    static QString getExistingDirectory(const QUrl &startDir = QUrl(),
                                        QWidget *parent = nullptr,
                                        const QString &caption = QString());

    /**
     * Creates a modal directory-selection dialog and returns the selected
     * directory or an empty string if none was chosen.
     * This version supports remote urls.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     * @return the url to an existing directory (local or remote).
     *
     * @see KFileWidget::KFileWidget()
     * @deprecated use QFileDialog::getExistingDirectoryUrl(parent, caption, startDir)
     */
    static QUrl getExistingDirectoryUrl(const QUrl &startDir = QUrl(),
                                        QWidget *parent = nullptr,
                                        const QString &caption = QString());

    /**
     * Creates a modal file dialog with an image previewer and returns the
     * selected url or an empty string if none was chosen.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param parent The widget the dialog will be centered on initially.
     * @param caption The name of the dialog widget.
     *
     * @see KFileWidget::KFileWidget()
     */
    static QUrl getImageOpenUrl(const QUrl &startDir = QUrl(),
                                QWidget *parent = nullptr,
                                const QString &caption = QString());

    /**
     * Sets the mode of the dialog.
     *
     * The mode is defined as (in kfile.h):
     * \code
     *    enum Mode {
     *         File         = 1,
     *         Directory    = 2,
     *         Files        = 4,
     *         ExistingOnly = 8,
     *         LocalOnly    = 16
     *    };
     * \endcode
     * You can OR the values, e.g.
     * \code
     * KFile::Modes mode = KFile::Files |
     *                     KFile::ExistingOnly |
     *                     KFile::LocalOnly );
     * setMode( mode );
     * \endcode
     */
    void setMode(KFile::Modes m);

    /**
     * Returns the mode of the filedialog.
     * @see setMode()
     */
    KFile::Modes mode() const;

    /**
     * Sets the text to be displayed in front of the selection.
     *
     * The default is "Location".
     * Most useful if you want to make clear what
     * the location is used for.
     */
    void setLocationLabel(const QString &text);

    /**
     * Returns the KFileWidget that implements most of this file dialog.
     * If you link to libkfile you can cast this to a KFileWidget*.
     */
    KFileWidget *fileWidget();

    /**
     * Returns a pointer to the toolbar.
     *
     * You can use this to insert custom
     * items into it, e.g.:
     * \code
     *      yourAction = new KAction( i18n("Your Action"), 0,
     *                                this, SLOT( yourSlot() ),
     *                                this, "action name" );
     *      yourAction->plug( kfileDialog->toolBar() );
     * \endcode
     */
    KToolBar *toolBar() const;

    /**
     * @returns a pointer to the OK-Button in the filedialog. You may use it
     * e.g. to set a custom text to it.
     */
    QPushButton *okButton() const;

    /**
     * @returns a pointer to the Cancel-Button in the filedialog. You may use
     * it e.g. to set a custom text to it.
     */
    QPushButton *cancelButton() const;

    /**
     * @returns the combobox used to type the filename or full location of the file.
     * You need to link to libkfile to use this widget.
     */
    KUrlComboBox *locationEdit() const;

    /**
     * @returns the combobox that contains the filters
     * You need to link to libkfile to use this widget.
     */
    KFileFilterCombo *filterWidget() const;

    /**
     * @returns a pointer to the action collection, holding all the used KActions.
     */
    KActionCollection *actionCollection() const;

    /**
     * This method implements the logic to determine the user's default directory
     * to be listed. E.g. the documents directory, home directory or a recently
     * used directory.
     *
     * @param startDir Starting directory or @c kfiledialog:/// URL.
     *                 Refer to the KFileWidget documentation for more information
     *                 on this parameter.
     * @param recentDirClass If the @c kfiledialog:/// syntax is used, this
     *        will return the string to be passed to KRecentDirs::dir() and
     *        KRecentDirs::add().
     * @return The URL that should be listed by default (e.g. by KFileDialog or
     *         KDirSelectDialog).
     *
     * @see KFileWidget::KFileWidget()
     * @see KFileWidget::getStartUrl(const QUrl& startDir, QString& recentDirClass);
     */
    static QUrl getStartUrl(const QUrl &startDir, QString &recentDirClass);

    /**
     * @internal
     * Used by KDirSelectDialog to share the dialog's start directory.
     */
    static void setStartDir(const QUrl &directory);

#ifdef Q_OS_WIN
public Q_SLOTS:
    int exec();
#endif

Q_SIGNALS:
    /**
      * Emitted when the user selects a file. It is only emitted in single-
      * selection mode. The best way to get notified about selected file(s)
      * is to connect to the accepted() signal inherited from QDialog
      * and call selectedFile(), selectedFiles(),
      * selectedUrl() or selectedUrls().
      *
      * \since 4.4
      */
    void fileSelected(const QUrl &);

    /**
      * Emitted when the user highlights a file.
      *
      * \since 4.4
      */
    void fileHighlighted(const QUrl &);

    /**
     * Emitted when the user hilights one or more files in multiselection mode.
     *
     * Note: fileHighlighted() or fileSelected() are @em not
     * emitted in multiselection mode. You may use selectedItems() to
     * ask for the current highlighted items.
     * @see fileSelected
     */
    void selectionChanged();

    /**
     * Emitted when the filter changed, i.e. the user entered an own filter
     * or chose one of the predefined set via setFilter().
     *
     * @param filter contains the new filter (only the extension part,
     * not the explanation), i.e. "*.cpp" or "*.cpp *.cc".
     *
     * @see setFilter()
     * @see currentFilter()
     */
    void filterChanged(const QString &filter);

protected:
    /**
     * Reimplemented to animate the cancel button.
     */
    void keyPressEvent(QKeyEvent *e) override;

    /**
     * Reimplemented for saving the dialog geometry.
     */
    void hideEvent(QHideEvent *event) override;

protected Q_SLOTS:
    virtual void slotOk();
    void accept() override;
    virtual void slotCancel();

private:
    Q_DISABLE_COPY(KFileDialog)

    KFileDialogPrivate *const d;
};

#endif
