
/*
 * examples/power_rank.C
 *
 * Copyright (C) 2005, 2010 J-G Dumas
 * ========LICENCE========
 * This file is part of the library LinBox.
 *
 * LinBox is free software: you can redistribute it and/or modify
 * it under the terms of the  GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * ========LICENCE========
 */

/** \file examples/power_rank.C
 * @example  examples/power_rank.C
  \brief Rank of sparse matrix over Z or Zp.
  \ingroup examples
  */
#include <linbox/linbox-config.h>

#include <iostream>

#include <givaro/modular.h>
#include <linbox/matrix/sparse-matrix.h>
#include <linbox/algorithms/smith-form-sparseelim-local.h>

using namespace LinBox;
using namespace std;


template<typename Base>
int tmain (int argc, char **argv)
{
	commentator().setMaxDetailLevel (-1);
	commentator().setMaxDepth (-1);
	commentator().setReportStream (std::cerr);

	ifstream input (argv[1]);
	if (!input) { cerr << "Error opening matrix file: " << argv[1] << endl; return -1; }

    Base p; Givaro::Caster(p,Givaro::Integer(argv[2]));
    Base q; Givaro::Caster(q,Givaro::Integer(argv[3]));
    typedef Givaro::Modular<Base> Field;
    Field F(q);
    MatrixStream<Field> ms( F, input );
    SparseMatrix<Field, SparseMatrixFormat::SparseSeq > B (ms);
    cout << "B is " << B.rowdim() << " by " << B.coldim() << endl;
    if (B.rowdim() <= 20 && B.coldim() <= 20) B.write(cout) << endl;

		// using Sparse Elimination
    PowerGaussDomain< Field > PGD( F );
    std::vector<std::pair<size_t,Base> > local;

    Givaro::Timer tq; tq.clear(); tq.start();
    PGD(local, B, q, p);
    tq.stop();


    F.write(std::cout << "Local Smith Form ") << " : " << std::endl << '(';
    for (auto ip = local.begin(); ip != local.end(); ++ip) 
        std::cout << ip->first << " " << ip->second << ", ";
    cout << ")" << endl;


    std::cerr << tq << std::endl;

	return 0;
}

int main(int argc, char ** argv) {
	if (argc < 4 || argc > 5) {	
        cerr << "Usage: rank <matrix-file-in-supported-format> <prime> <prime-power> [<method>]" << endl; return -1; }

    Givaro::Integer q(argv[3]);
    size_t method( argc>4? atoi(argv[4]) : 0);
    const size_t logq( (size_t)ceil(logtwo(q)) );
    
    if ( (method == 1) || ( (method==0) && (logq<63) ) ) {
        return tmain<int64_t>(argc,argv);
    } else {
        if ( (method == 2) ) {
            return tmain<Givaro::Integer>(argc,argv);
        } else {
            if (! method) method = (size_t)std::max(6.,  ceil(log(logq)/log(2.))  );
            switch (method) {
                case 6: return tmain<RecInt::ruint<6>>(argc,argv);
                case 7: return tmain<RecInt::ruint<7>>(argc,argv); 
                case 8: return tmain<RecInt::ruint<8>>(argc,argv);
                case 9: return tmain<RecInt::ruint<9>>(argc,argv);
                case 10: return tmain<RecInt::ruint<10>>(argc,argv);
                case 11: return tmain<RecInt::ruint<11>>(argc,argv);
            }
        }
    }
    
    return 0;
}


// Local Variables:
// mode: C++
// tab-width: 8
// indent-tabs-mode: nil
// c-basic-offset: 8
// End:
// vim:sts=8:sw=8:ts=8:noet:sr:cino=>s,f0,{0,g0,(0,\:0,t0,+0,=s
