#include <iostream>
#include <cstdlib>
#include <cmath>
#include <cassert>
#include <X11/extensions/Xrandr.h>

#include "applet.h"
#include "keyboard.h"
#include "switcher.h"
#include "slider.h"
#include "touchpad.h"
#include "logger.h"
#include "image_cache.h"
#include "display.h"
#include "vibrator.h"


using namespace std;



using namespace std;

struct Config {
    string layout_filename;
    string hor_layout_filename;
    int move_up_down;
    int vib_time;
    int vib_power;
    int window_width;
    int window_height;
    int hor_window_width;
    int hor_window_height;
    int touchpad_width;
    int touchpad_height;
    int hor_touchpad_width;
    int hor_touchpad_height;
    vector<vector<short int *> > kcolor;
    int background_r, background_g, background_b;
    int logocolor_r, logocolor_g, logocolor_b;
    int slider_height;
    int slider_width;
    int hor_slider_height;
    int hor_slider_width;
    int slider_dist;
    string font_path;
    int font_size;
    string apps_path;
    string icons_path;
    string fonts_png_path;
    double touchpad_sensivity;
    bool switcher_on;
    int switcher_height;
    int switcher_width;
};

Config config;

void ReadConfig(const string& filename, Config& cfg){
    ifstream is;
    is.open(filename.c_str());
    if (!is.good())
	LOG(FATAL) << "Failed to load config file: " << filename;
    string first;
    bool eatlast = false;
    cfg.kcolor = vector<vector<short int*> >(2);
    while (is.good()){
	if (!eatlast)
	    is >> first;
	else
	    eatlast = false;
	if (first == "layout_filename")
	    is >> cfg.layout_filename;
	if (first == "landscape_layout_filename")
	    is >> cfg.hor_layout_filename;
	if (first == "move_up_down")
	    is >> cfg.move_up_down;
	if (first == "vib_time")
	    is >> cfg.vib_time;
	if (first == "vib_power")
	    is >> cfg.vib_power;
	if (first == "window_width")
	    is >> cfg.window_width;
	if (first == "window_height")
	    is >> cfg.window_height;
	if (first == "landscape_window_width")
	    is >> cfg.hor_window_width;
	if (first == "landscape_window_height")
	    is >> cfg.hor_window_height;
	if (first == "touchpad_width")
	    is >> cfg.touchpad_width;
	if (first == "touchpad_height")
	    is >> cfg.touchpad_height;
	if (first == "landscape_touchpad_width")
	    is >> cfg.hor_touchpad_width;
	if (first == "landscape_touchpad_height")
	    is >> cfg.hor_touchpad_height;
	if (first == "background_color")
	    is >> cfg.background_r >> cfg.background_g >> cfg.background_b;
	if (first == "logo_color")
	    is >> cfg.logocolor_r >> cfg.logocolor_g >> cfg.logocolor_b;
	if (first == "slider_height")
	    is >> cfg.slider_height;
	if (first == "slider_width")
	    is >> cfg.slider_width;
	if (first == "landscape_slider_height")
	    is >> cfg.hor_slider_height;
	if (first == "landscape_slider_width")
	    is >> cfg.hor_slider_width;
	if (first == "slider_dist")
	    is >> cfg.slider_dist;
	if (first == "font_path")
	    is >> cfg.font_path;
	if (first == "font_size")
	    is >> cfg.font_size;
	if (first == "applications_path")
	    is >> cfg.apps_path;
	if (first == "icons_path")
	    is >> cfg.icons_path;
	if (first == "fonts_png_path")
	    is >> cfg.fonts_png_path;
	if (first == "touchpad_sensivity")
	    is >> cfg.touchpad_sensivity;

	if (first == "switcher_height")
	    is >> cfg.switcher_height;
	if (first == "switcher_width")
	    is >> cfg.switcher_width;

	if (first == "switcher_on"){
	    string tmp;
	    is >> tmp;
	    if (tmp == "yes")
		cfg.switcher_on = true;
	    else 
		cfg.switcher_on = false;
	}


	if (first == "key_color" || first == "trans_key_color"){
	    int ind = 1;
	    if (first == "trans_key_color")
		ind = 0;
	    is >> first;
	    while (is.good() && first == "color"){
		short int *rgb = new short int[3];
		is >> rgb[0] >> rgb[1] >> rgb[2];
		cfg.kcolor[ind].push_back(rgb);
		is >> first;
	    }
	    eatlast = true;
	}
    }
}



int main(int argc, char* argv[])
{
    string config_filename = "/etc/literki/literki.conf";
    if (argc>=2)
	config_filename = argv[1];
    LOG(INFO) << "Reading config file";
    ReadConfig(config_filename, config);

    MyDisplay display;
    if (!display.Open())
	LOG(FATAL) << "Failed to initialize!";

    // keyboard and launcher params
    AppletWindowParams params(config.window_width,
			      config.window_height,
			      config.hor_window_width,
			      config.hor_window_height,
			      display,
			      config.font_path,
			      config.font_size);

    AppletWindowParams slider_params(
	    config.slider_width,
	    config.slider_height,
	    config.hor_slider_width,
	    config.hor_slider_height,
	    display,
	    config.font_path,
	    config.font_size);

    AppletWindowParams switcher_params(
	    config.switcher_width,
	    config.switcher_height,
	    config.switcher_width,
	    config.switcher_height,
	    display,
	    config.font_path,
	    config.font_size);

    AppletWindowParams touchpad_params(
	    config.touchpad_width,
	    config.touchpad_height,
	    config.hor_touchpad_width,
	    config.hor_touchpad_height,
	    display,
	    config.font_path,
	    config.font_size);


    KeyboardForegroundColors key_colors;

    for (int i=0; i<config.kcolor[0].size(); ++i)
        key_colors.transparent.push_back(display.MakeXColor(config.kcolor[0][i]));
    for (int i=0; i<config.kcolor[1].size(); ++i)
        key_colors.opaque.push_back(display.MakeXColor(config.kcolor[1][i]));

 /*   LOG(INFO) << "Setting highlite window";
    MyWindow highlite(50, 50, display);
    highlite.Create(0, display.Height());
    highlite.SetBackground(display.MakeXColor(0,0,0));
    highlite.SetForeground(display.MakeXColor(255,255,255));
    highlite.SetFont(params.font_path);*/

    image_cache.SetImagesPath(config.fonts_png_path);

    Vibrator vibrator(config.vib_time, config.vib_power);

    LOG(INFO) << "Creating keyboard";
    Keyboard keyboard(params, config.hor_layout_filename, config.layout_filename, key_colors);
    LOG(INFO) << "Creating switcher";
    Switcher switcher(switcher_params, "apps_icon", "phone_icon", "launch_icon");
    switcher.SetBackground(display.MakeXColor(0,0,0));
    switcher.SetForeground(display.MakeXColor(200,200,200));
    keyboard.SetBackground(display.MakeXColor(0,0,0));
    keyboard.SetForeground(display.MakeXColor(255,255,255));
    keyboard.SetVibrator(&vibrator);
    
    LOG(INFO) << "Creating slider for keyboard";
    Slider slider(slider_params, config.slider_dist);
    slider.SetBackground(display.MakeXColor(0,0,1));
    slider.SetTransparent(true);
    slider.AddAction(Slider::SLIDE_N, "keyboard_show");
    slider.AddAction(Slider::SLIDE_E, "$win_ctrl1");
    slider.AddAction(Slider::SLIDE_W, "$win_ctrl2");

    LOG(INFO) << "Creating touchpad";
    Touchpad touchpad(touchpad_params, "pointer", config.touchpad_sensivity);
    touchpad.SetBackground(display.MakeXColor(0,0,0));
    touchpad.SetForeground(display.MakeXColor(255,255,255));
    touchpad.SetTransparent(true);
    touchpad.SetVibrator(&vibrator);

    LOG(INFO) << "Creating slider for touchpad";
    Slider slider2(slider_params, config.slider_dist);
    slider2.SetBackground(display.MakeXColor(0,0,1));
    slider2.SetTransparent(true);
    slider2.AddAction(Slider::SLIDE_N, "touchpad_show");

    LOG(INFO) << "Creating close window slider";
    Slider slider3(slider_params, config.slider_dist);
    slider3.SetBackground(display.MakeXColor(0,0,1));
    slider3.SetTransparent(true);
    slider3.AddAction(Slider::SLIDE_S, "$win_ctrl_close");

    LOG(INFO) << "Creating slider for switcher";
    Slider slider4(slider_params, config.slider_dist);
    slider4.SetBackground(display.MakeXColor(0,0,1));
    slider4.SetTransparent(true);
    slider4.AddAction(Slider::SLIDE_S, "switcher_show");

    LOG(INFO) << "Setting position and orientation";
    keyboard.SetPosition(Applet::SOUTH);
    slider.SetPosition(Applet::SOUTH_EAST);
    Applet::Rotation rot = (display.Width() <= 480) ? Applet::PORTRAIT : Applet::LANDSCAPE;
    keyboard.WindowRotate(rot);
    slider.WindowRotate(rot);
    touchpad.SetPosition(Applet::SOUTH);
    touchpad.WindowRotate(rot);
    touchpad.Hide();
    switcher.SetPosition(Applet::NORTH);
    switcher.WindowRotate(rot);
    slider2.SetPosition(Applet::SOUTH_WEST);
    slider2.WindowRotate(rot);
    slider3.SetPosition(Applet::NORTH_WEST);
    slider3.WindowRotate(rot);
    slider4.SetPosition(Applet::NORTH_EAST);
    slider4.WindowRotate(rot);


    LOG(INFO) << "Loading keyboard layout";
    LOG(INFO) << "Showing keyboard";
    keyboard.SetTransparent(true);
    keyboard.Hide();
    switcher.Hide();


    slider.Show();
    slider2.Show();
    slider3.Show();
    if (config.switcher_on)
	slider4.Show();
    else
	slider4.Hide();


    int bpressx=-1, bpressy=-1;

    vector<Applet*> applets;
    applets.push_back(&keyboard);
    applets.push_back(&slider);
    applets.push_back(&slider2);
    applets.push_back(&slider3);
    if (config.switcher_on)
	applets.push_back(&slider4);
    applets.push_back(&touchpad);
    applets.push_back(&switcher);

    LOG(INFO) << "Processing events";

    while (1){
	XEvent xevt;
        display.GetEvent(&xevt);

	AppAction action;
	
	if (xevt.type == display.randr_event_base + RRScreenChangeNotify){
	    XRRScreenChangeNotifyEvent* rev = (XRRScreenChangeNotifyEvent*) &xevt;
	    XRRUpdateConfiguration(&xevt);
	    LOG(INFO) << "Randr event ------------ " << display.Width() << " " << display.Height() << " => " << rev->width << " " << rev->height;
	    int prev_width = display.Width(),
		prev_height = display.Height();

	//    if (rev->width != display.Width()){

	    display.UpdateWidthHeight();
	    LOG(INFO) << "Updated res " << display.Width() << " " << display.Height();
	    if (display.Width() != prev_width || display.Height() != prev_height){
	//	display.SetWidth(rev->width);
	//	display.SetHeight(rev->height);
	    Applet::Rotation rot = (display.Width() <= 480) ? Applet::PORTRAIT : Applet::LANDSCAPE;
	    for (int i=0; i<applets.size(); ++i)
	        applets[i]->WindowRotate(rot);
	    }
	//    }
	    continue;
	}
	switch (xevt.type)
        {
        case Expose:
            if (xevt.xexpose.count != 0) continue;
	    for (int i=0; i<applets.size(); ++i)
		if (applets[i]->IsWindowOwner(xevt.xexpose.window)){
		    applets[i]->Refresh();
		    break;
		}
            break;

	case ButtonPress:
	    for (int i=0; i<applets.size(); ++i)
		if (applets[i]->IsWindowOwner(xevt.xbutton.window)){
		    action = applets[i]->OnMousePress(xevt.xbutton.x, xevt.xbutton.y);
		    break;
		}
	    break;

	case MotionNotify:
	    for (int i=0; i<applets.size(); ++i)
		if (applets[i]->IsWindowOwner(xevt.xmotion.window)){
		    action = applets[i]->OnMouseMove(xevt.xmotion.x, xevt.xmotion.y);
		    break;
		}
	    break;			  

	case ButtonRelease: {
	    for (int i=0; i<applets.size(); ++i)
		if (applets[i]->IsWindowOwner(xevt.xbutton.window)){
		    action = applets[i]->OnMouseRelease(xevt.xbutton.x, xevt.xbutton.y);
		    break;
		}
	    break;			  

	    }

        case ConfigureNotify:
			    break;

        }
	if (action == "slider_show"){
	    slider.Show();
	    slider2.Show();
	} else
	if (action == "top_slider_show"){
	    slider3.Show();
	    slider4.Show();
	} else
	if (action == "keyboard_show"){
	    keyboard.Show();
	    keyboard.Refresh();
	    slider.Hide();
	    slider2.Hide();
	} else
	if (action == "switcher_show"){
	    LOG(INFO) << "Switcher";
	    switcher.Show();
	    slider3.Hide();
	    slider4.Hide();
	} else
	if (action == "touchpad_show"){
	    touchpad.Show();
	    slider.Hide();
	    slider2.Hide();
	} else
	display.GenKeyEvent(action);

    }

    return 0;
}

