/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     sor.c                                                          */
/*                                                                          */
/* description:  SSOR-method for scalar and decoupled vector valued         */
/*               problems                                                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Univesitaet Bremen                                           */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Claus-Justus Heine                                           */
/*             Abteilung fuer Angewandte Mathematik                         */
/*             Alberta-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg im Breisgau, Germany                        */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

int ssor_s(DOF_MATRIX *a, const DOF_REAL_VEC *f, const DOF_SCHAR_VEC *bound,
	   DOF_REAL_VEC *u, REAL omega, REAL tol, int max_iter, int info)
{
  FUNCNAME("ssor_s");
  const REAL *fvec;
  REAL       *uvec;
  S_CHAR     *bvec;
  REAL       max = 0.0, omega1, accu, unew;
  int        i, j, jcol, iter, dim;
  MATRIX_ROW *row;

  fvec = f->vec;
  uvec = u->vec;
  bvec = bound ? bound->vec : nil;

  TEST_EXIT(a->row_fe_space == a->col_fe_space,
	    "Row and column FE_SPACEs don't match!\n");

  if (a->row_fe_space->admin->hole_count > 0)
    dof_compress(a->row_fe_space->mesh);

  if (omega <= 0  ||  omega > 2)
  {
    ERROR("omega %le not in (0,2], setting omega = 1.0\n", omega);
    omega = 1.0;
  }
  omega1   = 1.0 - omega;

  if (info >= 2)
  {
    MSG("omega = %.3lf, tol = %.3le, max_iter = %d\n",
	omega, tol, max_iter);
  }

  for (iter = 0; iter < max_iter; iter++)
  {
    max =  0.0;
    dim = u->fe_space->admin->size_used; 

    for (i = 0; i < dim; i++) 
    {
      if (!bvec || bvec[i] < DIRICHLET)
      {
	if (a->matrix_row[i])
	{
	  accu = 0.0;
	  for (row = a->matrix_row[i]; row; row = row->next) {
	    for (j=0; j<ROW_LENGTH; j++) {
	      jcol = row->col[j];
	      if (ENTRY_USED(jcol) && (jcol != i)) {
		accu += row->entry[j] * uvec[jcol];
	      }
	      else {
		if (jcol == NO_MORE_ENTRIES)
		  break;
	      }
	    }
	  }
	  if ((row = a->matrix_row[i]))
	  {
	    unew = omega1 * uvec[i] + omega * (fvec[i] - accu) / row->entry[0];
	  }
	  else
	  {
	    unew = 0.0;
	  }
	  max = MAX(max, ABS(uvec[i] - unew));
	  uvec[i] = unew;
	}
      }
    }

    for (i = dim-1; i >= 0; i--) 
    {
      if (!bvec || bvec[i] < DIRICHLET)
      {
	if (a->matrix_row[i])
	{
	  accu = 0.0;
	  for (row = a->matrix_row[i]; row; row = row->next) {
	    for (j=0; j<ROW_LENGTH; j++) {
	      jcol = row->col[j];
	      if (ENTRY_USED(jcol) && (jcol != i)) {
		accu += row->entry[j] * uvec[jcol];
	      }
	      else {
		if (jcol == NO_MORE_ENTRIES)
		  break;
	      }
	    }
	  }
	  if ((row = a->matrix_row[i]))
	  {
	    unew = omega1 * uvec[i] + omega * (fvec[i] - accu) / row->entry[0];
	  }
	  else
	  {
	    unew = 0.0;
	  }
	  max = MAX(max, ABS(uvec[i] - unew));
	  uvec[i] = unew;
	}
      }
    }
    if (info >= 4)
      MSG("iter %3d: max = %.3le\n",iter,max);
    
    if (max < tol) break;
  }
  
  if (info >= 2)
  {
    if (iter < max_iter)
      MSG("convergence after iter %3d: max = %.3le\n", iter, max);
    else
      MSG("NO CONVERGENCE after iter %3d: max = %.3le\n", iter, max);
  }
  return(iter);
}

int ssor_d(DOF_MATRIX *a, const DOF_REAL_D_VEC *f, const DOF_SCHAR_VEC *bound,
	   DOF_REAL_D_VEC *u, REAL omega, REAL tol, int max_iter, int info)
{
  FUNCNAME("ssor_d");
  const REAL_D *fvec;
  REAL_D       *uvec;
  S_CHAR       *bvec;
  REAL         max = 0.0, omega1;
  REAL_D       accu, unew;
  int          n, i, j, jcol, iter, dim;
  MATRIX_ROW *row;

  fvec = (const REAL_D *) f->vec;
  uvec = u->vec;
  bvec = bound ? bound->vec : nil;

  TEST_EXIT(a->row_fe_space == a->col_fe_space,
	    "Row and column FE_SPACEs don't match!\n");

  if (a->row_fe_space->admin->hole_count > 0)
    dof_compress(a->row_fe_space->mesh);

  if (omega <= 0  ||  omega > 2)
  {
    WARNING("omega %le not in (0,2], setting omega = 1.0\n", omega);
    omega = 1.0;
  }
  omega1   = 1.0 - omega;

  if (info >= 2)
  {
    MSG("omega = %.3lf, tol = %.3le, max_iter = %d\n",
	omega, tol, max_iter);
  }

  for (iter = 0; iter < max_iter; iter++)
  {
    max =  0.0;
    dim = u->fe_space->admin->size_used; 

    for (i = 0; i < dim; i++) 
    {
      if (!bvec || bvec[i] < DIRICHLET)
      {
	if (a->matrix_row[i])
	{
	  SET_DOW(0.0, accu);
	  for (row = a->matrix_row[i]; row; row = row->next) 
	  {
	    for (j=0; j<ROW_LENGTH; j++) 
	    {
	      jcol = row->col[j];
	      if (ENTRY_USED(jcol) && (jcol != i)) 
	      {
		for (n = 0; n < DIM_OF_WORLD; n++)
		  accu[n] += row->entry[j] * uvec[jcol][n];
	      }
	      else 
	      {
		if (jcol == NO_MORE_ENTRIES)
		  break;
	      }
	    }
	  }
	  if ((row = a->matrix_row[i]))
	  {
	    for (n = 0; n < DIM_OF_WORLD; n++)
	      unew[n] = (omega1*uvec[i][n] 
			 + omega*(fvec[i][n] -accu[n])/row->entry[0]);
	  }
	  else
	  {
	    SET_DOW(0.0, unew);
	  }

	  for (n = 0; n < DIM_OF_WORLD; n++)
	  {
	    max = MAX(max, ABS(uvec[i][n] - unew[n]));
	    uvec[i][n] = unew[n];
	  }
	}
      }
    }

    for (i = dim-1; i >= 0; i--) 
    {
      if (!bvec || bvec[i] < DIRICHLET)
      {
	if (a->matrix_row[i])
	{
	  SET_DOW(0.0, accu);
	  for (row = a->matrix_row[i]; row; row = row->next)
	  {
	    for (j=0; j<ROW_LENGTH; j++)
	    {
	      jcol = row->col[j];
	      if (ENTRY_USED(jcol) && (jcol != i))
	      {
		for (n = 0; n < DIM_OF_WORLD; n++)
		  accu[n] += row->entry[j]*uvec[jcol][n];
	      }
	      else
	      {
		if (jcol == NO_MORE_ENTRIES)
		  break;
	      }
	    }
	  }
	  if ((row = a->matrix_row[i]))
	  {
	    for (n = 0; n < DIM_OF_WORLD; n++)
	      unew[n] = (omega1*uvec[i][n] 
			 + omega*(fvec[i][n] - accu[n])/row->entry[0]);
	  }
	  else
	  {
	    SET_DOW(0.0, unew);
	  }
	  for (n = 0; n < DIM_OF_WORLD; n++)
	  {
	    max = MAX(max, ABS(uvec[i][n] - unew[n]));
	    uvec[i][n] = unew[n];
	  }
	}
      }
    }
    if (info >= 4)
      MSG("iter %3d: max = %.3le\n",iter,max);
    
    if (max < tol) break;
  }
  
  if (info >= 2)
  {
    if (iter < max_iter)
      MSG("convergence after iter %3d: max = %.3le\n", iter, max);
    else
      MSG("NO CONVERGENCE after iter %3d: max = %.3le\n", iter, max);
  }
  return(iter);
}

int ssor_dowb(DOF_DOWB_MATRIX *a,
	      const DOF_REAL_D_VEC *f, const DOF_SCHAR_VEC *bound,
	      DOF_REAL_D_VEC *u, REAL omega, REAL tol, int max_iter, int info)
{
  FUNCNAME("ssor_dowb");
  const REAL_D *fvec;
  REAL_D       *uvec;
  S_CHAR       *bvec;
  REAL         max = 0.0, omega1;
  REAL_D       accu, unew;
  int          n, i, j, jcol, iter = 0, dim;
  DOWB_MATRIX_ROW *row;

  fvec = (const REAL_D *) f->vec;
  uvec = u->vec;
  bvec = bound ? bound->vec : nil;

  TEST_EXIT(a->row_fe_space == a->col_fe_space,
	    "Row and column FE_SPACEs don't match!\n");

  if (a->row_fe_space->admin->hole_count > 0)
    dof_compress(a->row_fe_space->mesh);

  if (omega <= 0  ||  omega > 2)
  {
    WARNING("omega %le not in (0,2], setting omega = 1.0\n", omega);
    omega = 1.0;
  }
  omega1   = 1.0 - omega;

  if (info >= 2)
  {
    MSG("omega = %.3lf, tol = %.3le, max_iter = %d\n",
	omega, tol, max_iter);
  }

#undef MAT_BODY
#define MAT_BODY(F, C, P, S)						\
  for (iter = 0; iter < max_iter; iter++)				\
  {									\
    max =  0.0;								\
    dim = u->fe_space->admin->size_used; 				\
									\
    for (i = 0; i < dim; i++) 						\
    {									\
      if (!bvec || bvec[i] < DIRICHLET)					\
      {									\
	if (a->matrix_row[i])						\
	{								\
	  SET_DOW(0.0, accu);						\
	  for (row = a->matrix_row[i]; row; row = row->next) 		\
	  {								\
	    for (j=0; j<ROW_LENGTH; j++) 				\
	    {								\
	      jcol = row->col[j];					\
	      if (ENTRY_USED(jcol) && (jcol != i)) 			\
	      {								\
		F##V_DOW(C P row->entry.S[j], uvec[jcol], accu);	\
	      }								\
	      else 							\
	      {								\
		if (jcol == NO_MORE_ENTRIES)				\
		  break;						\
	      }								\
	    }								\
	  }								\
	  if ((row = a->matrix_row[i]))					\
	  {								\
	    AXPBY_DOW(omega, fvec[i], - omega, accu, unew);		\
	    AXPY_DOW(omega1, uvec[i], unew);				\
	    F##DIV_DOW(C P row->entry.S[0], unew, unew);		\
	  }								\
	  else								\
	  {								\
	    SET_DOW(0.0, unew);						\
	  }								\
									\
	  for (n = 0; n < DIM_OF_WORLD; n++)				\
	  {								\
	    max = MAX(max, ABS(uvec[i][n] - unew[n]));			\
	    uvec[i][n] = unew[n];					\
	  }								\
	}								\
      }									\
    }									\
									\
    for (i = dim-1; i >= 0; i--) 					\
    {									\
      if (!bvec || bvec[i] < DIRICHLET)					\
      {									\
	if (a->matrix_row[i])						\
	{								\
	  SET_DOW(0.0, accu);						\
	  for (row = a->matrix_row[i]; row; row = row->next)		\
	  {								\
	    for (j=0; j<ROW_LENGTH; j++)				\
	    {								\
	      jcol = row->col[j];					\
	      if (ENTRY_USED(jcol) && (jcol != i))			\
	      {								\
		F##V_DOW(C P row->entry.S[j], uvec[jcol], accu);	\
	      }								\
	      else							\
	      {								\
		if (jcol == NO_MORE_ENTRIES)				\
		  break;						\
	      }								\
	    }								\
	  }								\
	  if ((row = a->matrix_row[i]))					\
	  {								\
	    AXPBY_DOW(omega, fvec[i], - omega, accu, unew);		\
	    AXPY_DOW(omega1, uvec[i], unew);				\
	    F##DIV_DOW(C P row->entry.S[0], unew, unew);		\
	  }								\
	  else								\
	  {								\
	    SET_DOW(0.0, unew);						\
	  }								\
	  for (n = 0; n < DIM_OF_WORLD; n++)				\
	  {								\
	    max = MAX(max, ABS(uvec[i][n] - unew[n]));			\
	    uvec[i][n] = unew[n];					\
	  }								\
	}								\
      }									\
    }									\
    if (info >= 4)							\
      MSG("iter %3d: max = %.3le\n",iter,max);				\
    									\
    if (max < tol) break;						\
  }

  MAT_EMIT_BODY_SWITCH(a->type);

  if (info >= 2)
  {
    if (iter < max_iter)
      MSG("convergence after iter %3d: max = %.3le\n", iter, max);
    else
      MSG("NO CONVERGENCE after iter %3d: max = %.3le\n", iter, max);
  }
  return(iter);
}
