package org.jboss.cache.commands.write;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.Visitor;
import org.jboss.cache.notifications.event.NodeModifiedEvent;
import org.jboss.cache.optimistic.DataVersion;
import org.jboss.cache.transaction.GlobalTransaction;

import java.util.Collections;
import java.util.Map;

/**
 * Implements functionality defined by {@link org.jboss.cache.Cache#put(org.jboss.cache.Fqn, Object, Object)}.
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
public class PutKeyValueCommand extends AbstractVersionedDataCommand
{
   public static final int METHOD_ID = 3;
   public static final int VERSIONED_METHOD_ID = 39;

   private static final Log log = LogFactory.getLog(PutKeyValueCommand.class);
   private static boolean trace = log.isTraceEnabled();

   /* parametres */
   protected Object key;
   protected Object value;
   protected Object oldValue;

   public PutKeyValueCommand(GlobalTransaction gtx, Fqn fqn, Object key, Object value)
   {
      this.globalTransaction = gtx;
      this.fqn = fqn;
      this.key = key;
      this.value = value;
   }

   public PutKeyValueCommand()
   {
   }

   /**
    * Puts the specified key and value into the data map in the node referenced by the specified Fqn.
    *
    * @param ctx invocation context
    * @return the value being overwritten, if any, otherwise a null.
    */
   public Object perform(InvocationContext ctx)
   {
      if (trace)
      {
         log.trace(new StringBuilder("perform(").append(globalTransaction).append(", \"").
               append(fqn).append("\", k=").append(key).append(", v=").append(value).append(")"));
      }
      NodeSPI n = dataContainer.peekStrict(globalTransaction, fqn, false);
      if (notifier.shouldNotifyOnNodeModified())
      {
         notifier.notifyNodeModified(fqn, true, NodeModifiedEvent.ModificationType.PUT_DATA, n.getDataDirect(), ctx);
      }
      oldValue = n.putDirect(key, value);

      if (notifier.shouldNotifyOnNodeModified())
      {
         Map newData = Collections.singletonMap(key, value);
         notifier.notifyNodeModified(fqn, false, NodeModifiedEvent.ModificationType.PUT_DATA, newData, ctx);
      }
      return oldValue;
   }

   public void rollback()
   {
      NodeSPI n = dataContainer.peek(fqn, false, false);
      if (n == null) throw new CacheException("node " + fqn + " not found for rollback!");
      if (oldValue == null)
      {
         n.removeDirect(key);
      }
      else
      {
         n.putDirect(key, oldValue);
      }
   }

   public Object acceptVisitor(InvocationContext ctx, Visitor visitor) throws Throwable
   {
      return visitor.visitPutKeyValueCommand(ctx, this);
   }

   public Object getKey()
   {
      return key;
   }

   public Object getValue()
   {
      return value;
   }

   public void setKey(Object key)
   {
      this.key = key;
   }

   public void setValue(Object value)
   {
      this.value = value;
   }

   public int getCommandId()
   {
      if (isVersioned())
      {
         return VERSIONED_METHOD_ID;
      }
      else
      {
         return METHOD_ID;
      }
   }

   @Override
   public Object[] getParameters()
   {
      if (isVersioned())
         return new Object[]{globalTransaction, fqn, key, value, false, dataVersion};
      else
         return new Object[]{globalTransaction, fqn, key, value, false};
   }

   @Override
   public void setParameters(int commandId, Object[] args)
   {
      globalTransaction = (GlobalTransaction) args[0];
      fqn = (Fqn) args[1];
      key = args[2];
      value = args[3];
      if (isVersionedId(commandId)) dataVersion = (DataVersion) args[5];
   }

   @Override
   public boolean equals(Object o)
   {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;

      PutKeyValueCommand that = (PutKeyValueCommand) o;

      if (globalTransaction != null ? !globalTransaction.equals(that.globalTransaction) : that.globalTransaction != null)
         return false;
      if (key != null ? !key.equals(that.key) : that.key != null) return false;
      if (value != null ? !value.equals(that.value) : that.value != null) return false;

      return true;
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + (globalTransaction != null ? globalTransaction.hashCode() : 0);
      result = 31 * result + (key != null ? key.hashCode() : 0);
      result = 31 * result + (value != null ? value.hashCode() : 0);
      return result;
   }

   @Override
   protected boolean isVersionedId(int commandId)
   {
      return commandId == VERSIONED_METHOD_ID;
   }

   @Override
   public String toString()
   {
      return getClass().getSimpleName() + "{" +
            "fqn=" + fqn +
            ", dataVersion=" + dataVersion +
            ", globalTransaction=" + globalTransaction +
            ", key=" + key +
            ", value=" + value +
            ", oldValue=" + oldValue +
            '}';
   }
}
