/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2008 Ilsoo Byun, ETRI <widepis@etri.re.kr ,widepis@empal.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
################################################################################*/

#include "net_dev.h"

static const char hexval[16] = {
        '0', '1', '2', '3',
        '4', '5', '6', '7',
        '8', '9', 'A', 'B',
        'C', 'D', 'E', 'F'
};

char *trim(char *str);
struct net_proc parse(char* line, int len);
void fill_net_proc(struct net_proc* target, char* line);

char *trim(char *str) {
      char *ibuf = str, *obuf = str;
      int i = 0, cnt = 0;

      if (str) {
            for (ibuf = str; *ibuf && isspace(*ibuf); ++ibuf)
                  ;
            if (str != ibuf)
                  memmove(str, ibuf, ibuf - str);

            while (*ibuf) {
                  if (isspace(*ibuf) && cnt)
                        ibuf++;
                  else {
                        if (!isspace(*ibuf))
                              cnt = 0;
                        else {
                              *ibuf = ' ';
                              cnt = 1;
                        }
                        obuf[i++] = *ibuf++;
                  }
            }
            obuf[i] = (char) NULL;

            while (--i >= 0) {
                  if (!isspace(obuf[i]))
                        break;
            }
            obuf[++i] = (char) NULL;
      }
      return str;
}

int CN_getNetDevices(struct net_proc** result) {
	*result = (struct net_proc*) malloc(sizeof(struct net_proc)*100);
	struct net_proc* current = *result;

	const char* filename = "/proc/net/dev";
	FILE* ifp;
	ifp = fopen(filename, "r");
	if (ifp == NULL) {
		fprintf(stderr, "can't open %s\n", filename);
		return -1;
	}

	char* line = NULL;
	ssize_t readed = 0;
	size_t len = 0;
	unsigned int count = 0;
	while((readed = getline(&line, &len, ifp)) > 0) {
		if ((count++) < 2) continue;
		*current = parse(line, readed);
		++current;
	}
	current = NULL;

	if (line) free(line);
	fclose(ifp);

	return count > 1 ? count-2 : 0;
}

void CN_releaseNetDevices(struct net_proc** result, int count) {
	int i = 0;
	for (; i < count; ++i) {
		struct net_proc device = (*result)[i];
		if (device.name) free((void *) device.name);
	}
	free(*result);
}

struct net_proc parse(char* line, int len) {
	char* token = NULL;
	char* saveptr;
	int col_num = 0;
	struct net_proc result;
	while((token = strtok_r(line, ":", &saveptr)) != NULL) {
		switch (col_num) {
		case 0:
			result.name = strdup(trim(token));
			break;
		case 1:
			fill_net_proc(&result, token);
			break;
		}
		line = NULL;
		++col_num;
	}
	return result;
}

void fill_net_proc(struct net_proc* target, char* line) {
	char* token = NULL;
	char* saveptr;
        int col_num = 0;
	while((token = strtok_r(line, " ", &saveptr)) != NULL) {
		switch (col_num) {
		case 0:
			target->rx = atoi(token);
			break;
		case 8:
			target->tx = atoi(token);
			break;
		}

		line = NULL;
		++col_num;
	}
}

const char* CN_getMacAddress(const char* name) {
        int fd;
        struct ifreq ifr;

        fd = socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL));
        if (fd < 0) {
                fprintf(stderr, "socket(2) failed: %s\n", strerror(errno));
                return NULL;
        }

        strcpy(ifr.ifr_name, "eth0");
        if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0) {
                fprintf(stderr, "ioctl(2) failed: %s\n", strerror(errno));
		close(fd);
                return NULL;
        }

        char* result = (char *) malloc(sizeof(char)*18);
	//memset(result, (int) NULL, sizeof(char)*16);
        char *hwaddr;
        hwaddr = ifr.ifr_hwaddr.sa_data;
        int i = 0;
        int first_char_idx = 0;
        for (;i < 6; ++i) {
                result[first_char_idx] = hexval[(hwaddr[i] >> 4) & 0x0F];
                result[first_char_idx+1] = hexval[hwaddr[i] & 0x0F];
                if (i != 5) result[first_char_idx+2] = ':';
                first_char_idx += 3;
        }
	if (i == 6)
		result[first_char_idx-1] = (char) NULL;

	close(fd);
        return result;
}

const char* CN_getIP(const char* name) {
        int fd;
        struct ifreq ifr;
        struct sockaddr_in *aptr;

        fd = socket(AF_INET, SOCK_DGRAM, 0);
        if (fd < 0) {
                fprintf(stderr, "socket(2) failed: %s\n", strerror(errno));
                return NULL;
        }

        strncpy(ifr.ifr_name, name, IFNAMSIZ);

        if (ioctl(fd, SIOCGIFADDR, &ifr) < 0) {
                fprintf(stderr, "ioctl(2) failed: %s\n", strerror(errno));
		close(fd);
                return NULL;
        }

        char* address = (char *) malloc(sizeof(char)*ADDR_MAX);
        aptr = (struct sockaddr_in *)&ifr.ifr_addr;
        inet_ntop(AF_INET, &aptr->sin_addr, address, ADDR_MAX);

	close(fd);
        return address;
}

const char* CN_getNetMask(const char* name) {
        int fd;
        struct ifreq ifr;
        struct sockaddr_in *aptr;

        fd = socket(AF_INET, SOCK_DGRAM, 0);
        if (fd < 0) {
                fprintf(stderr, "socket(2) failed: %s\n", strerror(errno));
                return NULL;
        }

        strncpy(ifr.ifr_name, name, IFNAMSIZ);

        if (ioctl(fd, SIOCGIFNETMASK, &ifr) < 0) {
                fprintf(stderr, "ioctl(2) failed: %s\n", strerror(errno));
		close(fd);
                return NULL;
        }

        char* address = (char *) malloc(sizeof(char)*ADDR_MAX);
        aptr = (struct sockaddr_in *)&ifr.ifr_netmask;
        inet_ntop(AF_INET, &aptr->sin_addr, address, ADDR_MAX);

	close(fd);
        return address;
}

const char* CN_getBroadcastAddress(const char* name) {
        int fd;
        struct ifreq ifr;
        struct sockaddr_in *aptr;

        fd = socket(AF_INET, SOCK_DGRAM, 0);
        if (fd < 0) {
                fprintf(stderr, "socket(2) failed: %s\n", strerror(errno));
                return NULL;
        }

        strncpy(ifr.ifr_name, name, IFNAMSIZ);

        if (ioctl(fd, SIOCGIFBRDADDR, &ifr) < 0) {
                fprintf(stderr, "ioctl(2) failed: %s\n", strerror(errno));
		close(fd);
                return NULL;
        }

        char* address = (char *) malloc(sizeof(char)*ADDR_MAX);
        aptr = (struct sockaddr_in *)&ifr.ifr_broadaddr;
        inet_ntop(AF_INET, &aptr->sin_addr, address, ADDR_MAX);

	close(fd);
        return address;
}

