!
!  aperture photometry
!
!  Copyright © 2010-18 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.


program aphot

  use fitsio
  use fitsaphot
  use mdaofotometr

  implicit none

  character(len=4*FLEN_FILENAME) :: record,key,val
  character(len=FLEN_FILENAME) :: filename,output
  real, dimension(:), allocatable :: raper
  real, dimension(2) :: ring
  logical :: verbose = .false., plog = .false.
  logical :: ex, exitus = .true.
  integer :: i,eq,naper

  ! we are predefine radii of apertures as logarithmic spiral,
  ! pitch 9.84 deg, raper(1) = 1, .. raper(12) = 20
  naper = 12
  allocate(raper(naper))
  forall( i = 1:size(raper) )
     raper(i) = exp(0.17338*((i-1)*1.570796327))
  end forall
  ring = (/20.0, 30.0/)

  do
     read(*,'(a)',end=20) record

     eq = index(record,'=')
     if( eq == 0 ) stop 'Malformed input record.'
     key = record(:eq-1)
     val = record(eq+1:)

     if( key == 'VERBOSE' ) then

        read(val,*) verbose

     else if( key == 'PIPELOG' ) then

        read(val,*) plog

     else if( key == 'NAPERTURES' ) then

        read(val,*) naper
        deallocate(raper)
        allocate(raper(naper))

     else if( key == 'APERTURES' ) then

        read(val,*) raper

     else if( key == 'RING' ) then

        read(val,*) ring

     else if( key == 'FILE' ) then

        read(val,*) filename, output

        if( verbose ) write(*,*) trim(filename)//": "

        call the_aphot(ex)
        exitus = exitus .and. ex

     end if

  end do

20 continue

  deallocate(raper)

  if( exitus ) then
     stop 0
  else
     stop 'An error during aperture photometry occurred.'
  end if

contains

  subroutine the_aphot(exitus)

    logical, intent(out) :: exitus

    real, dimension(:,:), allocatable :: data,stderr
    real, dimension(:), allocatable :: xcens,ycens,hstar,sky,sky_err
    real, dimension(:,:), allocatable :: apcts,apcts_err
    real :: lobad,hibad,fwhm,hwhm
    integer :: nrows, status

    status = 0
    call fits_aphot_read(filename,data,stderr,xcens,ycens,hstar, &
         lobad,hibad,fwhm,status)
    if( status /= 0 ) goto 666

    nrows = size(xcens)
    allocate(apcts(nrows,naper),apcts_err(nrows,naper), &
         sky(nrows),sky_err(nrows))

    call daophotsb(data,stderr,xcens,ycens,hstar,raper,ring,lobad,hibad,1.0, &
         verbose,plog,apcts,apcts_err,sky,sky_err,status)
    if( status /= 0 ) goto 666

    ! estimate of width parameter for Gaussian
    call estim_hwhm(data,xcens,ycens,sky,fwhm,lobad,hibad,hwhm)
    if( hwhm < 0 ) hwhm = fwhm / (2*sqrt(2*log(2.0)))

    call fits_aphot_save(filename, output, hwhm, raper, ring, &
         xcens, ycens, apcts,apcts_err,sky,sky_err, status)

666 continue

    exitus = status == 0

    if( allocated(data) ) deallocate(data,stderr)
    if( allocated(apcts) ) deallocate(xcens,ycens,hstar,apcts,apcts_err, &
         sky,sky_err)

  end subroutine the_aphot

end program aphot
