/**
 * @file
 * Test code for IMAP Message Sets
 *
 * @authors
 * Copyright (C) 2023 Richard Russon <rich@flatcap.org>
 *
 * @copyright
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define TEST_NO_MAIN
#include "config.h"
#include "acutest.h"
#include "mutt/lib.h"
#include "imap/msg_set.h"
#include "imap/private.h"
#include "test_common.h"

extern int ImapMaxCmdlen;

struct TestCase
{
  int input[150];
  const char *result;
};

struct Buffer *imap_exec_results = NULL;
bool imap_exec_fail = false;

int imap_exec(struct ImapAccountData *adata, const char *cmdstr, ImapCmdFlags flags)
{
  if (imap_exec_fail)
    return -1;

  buf_add_printf(imap_exec_results, "%s\n", cmdstr);
  return 0;
}

void test_sort(void)
{
  TEST_CASE("sort");
  {
    static const int data[] = {
      5,  49, 22, 44, 47, 41, 18, 3,  25, 1,  42, 12, 10, 11, 0,  43, 35,
      16, 28, 7,  6,  29, 27, 48, 21, 33, 19, 8,  31, 36, 37, 45, 34, 17,
      24, 4,  46, 39, 9,  40, 32, 14, 26, 30, 2,  20, 13, 15, 38, 23,
    };

    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    for (int i = 0; i < mutt_array_size(data); i++)
    {
      ARRAY_ADD(&uida, data[i]);
    }

    ARRAY_SORT(&uida, imap_sort_uid, NULL);

    for (int i = 0; i < mutt_array_size(data); i++)
    {
      unsigned int u = *ARRAY_GET(&uida, i);
      TEST_CHECK(u == i);
    }

    ARRAY_FREE(&uida);
  }
}

void test_make_degenerate(void)
{
  TEST_CASE("make (degenerate)");

  {
    // Missing uida
    int pos = 0;
    struct Buffer *msg_set = buf_pool_get();
    int rc = imap_make_msg_set(NULL, msg_set, &pos);
    TEST_CHECK(rc == 0);
    buf_pool_release(&msg_set);
  }

  {
    // Missing msg_set
    int pos = 0;
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    int rc = imap_make_msg_set(&uida, NULL, &pos);
    TEST_CHECK(rc == 0);
    ARRAY_FREE(&uida);
  }

  {
    // Missing pos
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    struct Buffer *msg_set = buf_pool_get();
    int rc = imap_make_msg_set(&uida, msg_set, NULL);
    TEST_CHECK(rc == 0);
    buf_pool_release(&msg_set);
    ARRAY_FREE(&uida);
  }

  {
    // Empty array
    int pos = 0;
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    struct Buffer *msg_set = buf_pool_get();
    int rc = imap_make_msg_set(&uida, msg_set, &pos);
    TEST_CHECK(rc == 0);
    buf_pool_release(&msg_set);
    ARRAY_FREE(&uida);
  }

  {
    // Out of range
    int pos = 10;
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    ARRAY_ADD(&uida, 1);
    ARRAY_ADD(&uida, 2);
    ARRAY_ADD(&uida, 3);
    struct Buffer *msg_set = buf_pool_get();
    int rc = imap_make_msg_set(&uida, msg_set, &pos);
    TEST_CHECK(rc == 0);
    buf_pool_release(&msg_set);
    ARRAY_FREE(&uida);
  }
}

void test_make_simple(void)
{
  TEST_CASE("make (simple)");

  /* Test single UIDs and runs of UIDs,
   * test message starting and ending with singles or runs,
   * test transitions between singles and runs. */
  static const struct TestCase tests[] = {
    // clang-format off
    { { 1 },                             "1" },
    { { 5 },                             "5" },
    { { 1, 2 },                          "1:2" },
    { { 1, 2, 3 },                       "1:3" },
    { { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 }, "1:10" },
    { { 1, 3, 5, 7, 9 },                 "1,3,5,7,9" },
    { { 2, 4, 8, 16 },                   "2,4,8,16" },
    { { 1, 4, 5, 6, 8 },                 "1,4:6,8" },
    { { 4, 5, 6, 10, 12, 13 },           "4:6,10,12:13" },
    { { 1, 2, 3, 6, 10, 11, 12 },        "1:3,6,10:12" },
    { { 1, 2, 3, 7, 8, 9, 20, 21, 22 },  "1:3,7:9,20:22" },
    // clang-format on
  };

  for (int i = 0; i < mutt_array_size(tests); i++)
  {
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    const struct TestCase *test = &tests[i];
    for (int j = 0; test->input[j] > 0; j++)
    {
      ARRAY_ADD(&uida, test->input[j]);
    }

    int pos = 0;
    struct Buffer *msg_set = buf_pool_get();
    int rc = imap_make_msg_set(&uida, msg_set, &pos);

    TEST_CHECK(rc == ARRAY_SIZE(&uida));
    TEST_CHECK_STR_EQ(buf_string(msg_set), test->result);

    buf_pool_release(&msg_set);
    ARRAY_FREE(&uida);
  }
}

void test_make_curated(void)
{
  TEST_CASE("make (curated)");

  // 50 Random, but manually-checked examples
  static const struct TestCase tests[] = {
    // clang-format off
    { { 1000,1003,1004,1005,1006,1007,1008,1009,1011,1012,1013,1014,1015,1016,1017,1018,1019,1021,
        1022,1023,1024,1025,1027,1028,1029,1030,1032,1033,1034,1036,1037,1038,1039,1040,1042,1044,1046,1049 },
        "1000,1003:1009,1011:1019,1021:1025,1027:1030,1032:1034,1036:1040,1042,1044,1046,1049" },

    { { 1000,1001,1003,1004,1006,1007,1009,1011,1012,1014,1016,1019,1021,1022,1023,1024,1026,1028,
        1029,1030,1031,1032,1033,1035,1038,1039,1040,1041,1042,1043,1046,1047,1048,1049 },
        "1000:1001,1003:1004,1006:1007,1009,1011:1012,1014,1016,1019,1021:1024,1026,1028:1033,1035,1038:1043,1046:1049" },

    { { 1000,1002,1003,1004,1005,1006,1007,1010,1011,1012,1013,1014,1016,1017,1018,1019,1020,1021,
        1022,1026,1029,1030,1031,1032,1034,1035,1036,1037,1038,1040,1041,1043,1044,1045,1046,1047,1049 },
        "1000,1002:1007,1010:1014,1016:1022,1026,1029:1032,1034:1038,1040:1041,1043:1047,1049" },

    { { 1000,1001,1002,1003,1004,1005,1006,1007,1009,1010,1011,1012,1014,1017,1018,1022,1025,1026,
        1028,1029,1030,1031,1033,1034,1035,1037,1040,1041,1042,1044,1046,1047,1048,1049 },
        "1000:1007,1009:1012,1014,1017:1018,1022,1025:1026,1028:1031,1033:1035,1037,1040:1042,1044,1046:1049" },

    { { 1000,1002,1003,1004,1005,1006,1007,1009,1010,1011,1012,1013,1014,1015,1016,1018,1020,1023,
        1024,1026,1027,1028,1029,1033,1035,1036,1037,1039,1040,1043,1044,1045,1046,1048,1049 },
        "1000,1002:1007,1009:1016,1018,1020,1023:1024,1026:1029,1033,1035:1037,1039:1040,1043:1046,1048:1049" },

    { { 1000,1001,1002,1003,1004,1005,1009,1010,1011,1012,1013,1014,1016,1019,1020,1022,1023,1024,
        1025,1026,1027,1028,1029,1030,1032,1033,1035,1037,1038,1039,1042,1044,1045,1046,1048,1049 },
        "1000:1005,1009:1014,1016,1019:1020,1022:1030,1032:1033,1035,1037:1039,1042,1044:1046,1048:1049" },

    { { 1000,1001,1002,1004,1005,1006,1007,1009,1010,1013,1014,1017,1018,1020,1021,1022,1023,1024,
        1025,1026,1027,1028,1033,1034,1035,1036,1037,1038,1039,1040,1041,1042,1043,1044,1045,1046,1047,1049 },
        "1000:1002,1004:1007,1009:1010,1013:1014,1017:1018,1020:1028,1033:1047,1049" },

    { { 1000,1001,1003,1004,1005,1006,1008,1009,1010,1012,1013,1015,1016,1018,1020,1021,1022,1023,
        1025,1027,1028,1030,1032,1033,1034,1035,1036,1037,1038,1039,1040,1041,1042,1043,1044,1045,1046,1048,1049 },
        "1000:1001,1003:1006,1008:1010,1012:1013,1015:1016,1018,1020:1023,1025,1027:1028,1030,1032:1046,1048:1049" },

    { { 1000,1004,1006,1008,1010,1015,1016,1017,1018,1019,1023,1024,1025,1026,1027,1029,1030,1032,
        1033,1034,1035,1037,1038,1039,1040,1041,1043,1045,1047,1049 },
        "1000,1004,1006,1008,1010,1015:1019,1023:1027,1029:1030,1032:1035,1037:1041,1043,1045,1047,1049" },

    { { 1000,1003,1005,1006,1007,1008,1009,1011,1013,1014,1016,1018,1019,1021,1024,1025,1026,1027,
        1029,1033,1034,1035,1037,1038,1039,1040,1042,1043,1045,1046,1047,1048 },
        "1000,1003,1005:1009,1011,1013:1014,1016,1018:1019,1021,1024:1027,1029,1033:1035,1037:1040,1042:1043,1045:1048" },

    { { 1000,1002,1003,1004,1005,1006,1007,1008,1009,1010,1011,1013,1015,1016,1018,1020,1021,1023,
        1024,1025,1027,1028,1029,1031,1032,1033,1036,1039,1041,1042,1043,1044,1046,1047,1048 },
        "1000,1002:1011,1013,1015:1016,1018,1020:1021,1023:1025,1027:1029,1031:1033,1036,1039,1041:1044,1046:1048" },

    { { 1000,1003,1004,1005,1006,1007,1008,1009,1010,1011,1012,1013,1014,1015,1018,1019,1020,1021,
        1022,1023,1024,1025,1026,1027,1028,1029,1033,1034,1035,1036,1038,1039,1040,1041,1042,1045,1046,1049 },
        "1000,1003:1015,1018:1029,1033:1036,1038:1042,1045:1046,1049" },

    { { 1000,1003,1005,1007,1008,1009,1010,1011,1012,1014,1019,1021,1022,1023,1024,1025,1026,1027,
        1028,1029,1030,1031,1035,1036,1039,1040,1041,1044,1045,1046,1047,1048,1049 },
        "1000,1003,1005,1007:1012,1014,1019,1021:1031,1035:1036,1039:1041,1044:1049" },

    { { 1000,1001,1004,1005,1007,1008,1009,1010,1011,1013,1015,1016,1017,1019,1020,1021,1022,1023,
        1024,1025,1026,1027,1028,1029,1030,1031,1032,1033,1035,1037,1038,1039,1040,1043,1045,1047,1048,1049 },
        "1000:1001,1004:1005,1007:1011,1013,1015:1017,1019:1033,1035,1037:1040,1043,1045,1047:1049" },

    { { 1000,1002,1003,1005,1007,1008,1009,1011,1012,1015,1017,1018,1019,1020,1021,1024,1025,1026,
        1027,1028,1030,1033,1034,1038,1039,1040,1041,1042,1043,1044,1045,1046,1047,1048 },
        "1000,1002:1003,1005,1007:1009,1011:1012,1015,1017:1021,1024:1028,1030,1033:1034,1038:1048" },

    { { 1000,1002,1003,1004,1005,1006,1007,1009,1010,1011,1012,1013,1017,1018,1019,1020,1021,1022,
        1023,1024,1025,1026,1027,1028,1030,1031,1032,1033,1034,1036,1037,1038,1039,1040,1041,1043,1044,1045,1047,1048,1049 },
        "1000,1002:1007,1009:1013,1017:1028,1030:1034,1036:1041,1043:1045,1047:1049" },

    { { 1000,1002,1004,1005,1006,1009,1011,1014,1015,1016,1018,1021,1023,1024,1025,1027,1028,1029,
        1030,1032,1033,1035,1036,1037,1039,1041,1043,1044,1045,1046,1048,1049 },
        "1000,1002,1004:1006,1009,1011,1014:1016,1018,1021,1023:1025,1027:1030,1032:1033,1035:1037,1039,1041,1043:1046,1048:1049" },

    { { 1000,1002,1003,1004,1005,1007,1009,1010,1011,1013,1015,1016,1018,1019,1020,1021,1022,1025,
        1027,1028,1030,1035,1036,1037,1038,1040,1041,1043,1045,1046,1047,1048,1049 },
        "1000,1002:1005,1007,1009:1011,1013,1015:1016,1018:1022,1025,1027:1028,1030,1035:1038,1040:1041,1043,1045:1049" },

    { { 1002,1005,1006,1007,1009,1010,1012,1013,1014,1016,1017,1018,1020,1022,1025,1027,1028,1029,
        1032,1033,1034,1035,1037,1038,1042,1043,1044,1045,1046,1047,1048,1049 },
        "1002,1005:1007,1009:1010,1012:1014,1016:1018,1020,1022,1025,1027:1029,1032:1035,1037:1038,1042:1049" },

    { { 1000,1003,1004,1008,1009,1013,1014,1015,1017,1018,1020,1021,1022,1023,1024,1025,1026,1029,
        1030,1031,1032,1034,1035,1038,1039,1040,1041,1043,1044,1046,1049 },
        "1000,1003:1004,1008:1009,1013:1015,1017:1018,1020:1026,1029:1032,1034:1035,1038:1041,1043:1044,1046,1049" },

    { { 1002,1004,1005,1007,1008,1012,1013,1016,1018,1019,1020,1021,1023,1024,1026,1027,1028,1029,
        1031,1032,1033,1036,1037,1038,1039,1041,1042,1043,1044,1046,1047,1049 },
        "1002,1004:1005,1007:1008,1012:1013,1016,1018:1021,1023:1024,1026:1029,1031:1033,1036:1039,1041:1044,1046:1047,1049" },

    { { 1000,1001,1002,1003,1005,1007,1011,1013,1015,1018,1022,1024,1025,1026,1027,1028,1029,1030,
        1031,1033,1034,1035,1037,1038,1041,1044,1045,1046,1047,1048,1049 },
        "1000:1003,1005,1007,1011,1013,1015,1018,1022,1024:1031,1033:1035,1037:1038,1041,1044:1049" },

    { { 1000,1001,1002,1003,1004,1006,1007,1008,1010,1011,1014,1016,1019,1020,1024,1025,1026,1027,
        1028,1029,1030,1033,1034,1036,1039,1041,1042,1044,1045,1046,1047,1048 },
        "1000:1004,1006:1008,1010:1011,1014,1016,1019:1020,1024:1030,1033:1034,1036,1039,1041:1042,1044:1048" },

    { { 1000,1002,1006,1008,1009,1011,1014,1016,1018,1019,1020,1023,1025,1026,1027,1028,1029,1031,
        1033,1035,1036,1037,1038,1039,1040,1041,1042,1043,1044,1045,1047,1048,1049 },
        "1000,1002,1006,1008:1009,1011,1014,1016,1018:1020,1023,1025:1029,1031,1033,1035:1045,1047:1049" },

    { { 1000,1001,1004,1006,1007,1008,1011,1012,1013,1016,1018,1019,1020,1021,1024,1025,1027,1028,
        1029,1030,1032,1033,1034,1035,1036,1037,1038,1039,1040,1041,1042,1044,1046,1047,1048 },
        "1000:1001,1004,1006:1008,1011:1013,1016,1018:1021,1024:1025,1027:1030,1032:1042,1044,1046:1048" },

    { { 1000,1001,1003,1004,1005,1006,1008,1009,1010,1011,1012,1013,1015,1016,1017,1018,1019,1022,
        1023,1025,1026,1031,1032,1033,1035,1037,1038,1039,1041,1043,1044,1045,1046,1047,1049 },
        "1000:1001,1003:1006,1008:1013,1015:1019,1022:1023,1025:1026,1031:1033,1035,1037:1039,1041,1043:1047,1049" },

    { { 1001,1002,1003,1004,1005,1006,1007,1010,1012,1014,1015,1019,1020,1021,1022,1023,1024,1025,
        1028,1029,1030,1033,1035,1036,1037,1038,1039,1040,1041,1043,1045,1047,1048 },
        "1001:1007,1010,1012,1014:1015,1019:1025,1028:1030,1033,1035:1041,1043,1045,1047:1048" },

    { { 1001,1005,1006,1007,1009,1010,1012,1013,1014,1015,1016,1017,1018,1019,1020,1021,1022,1023,
        1024,1025,1026,1027,1029,1030,1031,1033,1034,1036,1037,1038,1041,1043,1046,1049 },
        "1001,1005:1007,1009:1010,1012:1027,1029:1031,1033:1034,1036:1038,1041,1043,1046,1049" },

    { { 1000,1001,1003,1005,1006,1007,1008,1010,1012,1013,1015,1016,1018,1019,1022,1024,1026,1028,
        1030,1032,1033,1036,1037,1038,1041,1042,1043,1044,1045,1046,1048 },
        "1000:1001,1003,1005:1008,1010,1012:1013,1015:1016,1018:1019,1022,1024,1026,1028,1030,1032:1033,1036:1038,1041:1046,1048" },

    { { 1000,1004,1005,1006,1007,1009,1010,1011,1012,1013,1015,1017,1018,1019,1020,1021,1023,1027,
        1030,1032,1033,1035,1036,1037,1038,1039,1040,1042,1044,1045,1046,1047 },
        "1000,1004:1007,1009:1013,1015,1017:1021,1023,1027,1030,1032:1033,1035:1040,1042,1044:1047" },

    { { 1000,1001,1003,1004,1006,1007,1008,1009,1013,1016,1023,1025,1026,1027,1028,1029,1030,1032,
        1033,1034,1035,1036,1037,1039,1040,1042,1043,1044,1046,1047,1048 },
        "1000:1001,1003:1004,1006:1009,1013,1016,1023,1025:1030,1032:1037,1039:1040,1042:1044,1046:1048" },

    { { 1000,1002,1003,1004,1005,1007,1008,1009,1016,1017,1018,1019,1021,1022,1023,1024,1025,1026,
        1028,1030,1033,1034,1035,1038,1039,1041,1043,1044,1046,1047,1049 },
        "1000,1002:1005,1007:1009,1016:1019,1021:1026,1028,1030,1033:1035,1038:1039,1041,1043:1044,1046:1047,1049" },

    { { 1000,1001,1002,1006,1009,1010,1011,1012,1013,1014,1015,1016,1017,1018,1019,1020,1024,1026,
        1027,1029,1030,1031,1032,1033,1036,1037,1038,1041,1042,1043,1044,1045,1046,1047,1048,1049 },
        "1000:1002,1006,1009:1020,1024,1026:1027,1029:1033,1036:1038,1041:1049" },

    { { 1000,1001,1003,1004,1005,1011,1012,1015,1016,1018,1020,1023,1024,1025,1026,1028,1030,1031,
        1035,1036,1037,1038,1040,1041,1042,1044,1045,1048,1049 },
        "1000:1001,1003:1005,1011:1012,1015:1016,1018,1020,1023:1026,1028,1030:1031,1035:1038,1040:1042,1044:1045,1048:1049" },

    { { 1000,1001,1002,1003,1007,1008,1011,1012,1013,1014,1015,1017,1018,1021,1022,1023,1024,1025,
        1029,1032,1034,1035,1037,1038,1039,1040,1041,1044,1045,1047,1048,1049 },
        "1000:1003,1007:1008,1011:1015,1017:1018,1021:1025,1029,1032,1034:1035,1037:1041,1044:1045,1047:1049" },

    { { 1001,1002,1004,1005,1006,1009,1010,1011,1012,1013,1015,1017,1018,1020,1021,1024,1026,1027,
        1029,1030,1032,1034,1035,1036,1037,1038,1039,1040,1041,1043,1044,1045,1046,1047,1049 },
        "1001:1002,1004:1006,1009:1013,1015,1017:1018,1020:1021,1024,1026:1027,1029:1030,1032,1034:1041,1043:1047,1049" },

    { { 1001,1002,1005,1006,1008,1010,1011,1013,1014,1015,1016,1017,1018,1022,1023,1024,1025,1026,
        1028,1029,1030,1031,1032,1033,1034,1035,1036,1037,1038,1040,1041,1042,1044,1045,1046,1047,1048 },
        "1001:1002,1005:1006,1008,1010:1011,1013:1018,1022:1026,1028:1038,1040:1042,1044:1048" },

    { { 1002,1003,1004,1005,1006,1007,1008,1009,1010,1011,1013,1015,1017,1018,1019,1020,1021,1022,
        1023,1024,1025,1026,1027,1029,1030,1031,1033,1034,1035,1036,1038,1039,1040,1041,1042,1043,1046,1047,1049 },
        "1002:1011,1013,1015,1017:1027,1029:1031,1033:1036,1038:1043,1046:1047,1049" },

    { { 1000,1001,1002,1004,1006,1007,1009,1010,1012,1013,1014,1015,1016,1018,1021,1024,1026,1027,
        1028,1029,1033,1036,1038,1040,1041,1042,1047,1048,1049 },
        "1000:1002,1004,1006:1007,1009:1010,1012:1016,1018,1021,1024,1026:1029,1033,1036,1038,1040:1042,1047:1049" },

    { { 1000,1002,1004,1006,1008,1009,1010,1012,1013,1015,1016,1017,1020,1021,1022,1023,1026,1027,
        1030,1031,1034,1036,1038,1043,1044,1047,1048,1049 },
        "1000,1002,1004,1006,1008:1010,1012:1013,1015:1017,1020:1023,1026:1027,1030:1031,1034,1036,1038,1043:1044,1047:1049" },

    { { 1002,1003,1004,1005,1007,1008,1009,1010,1013,1014,1015,1016,1017,1018,1019,1020,1021,1022,
        1023,1025,1026,1028,1029,1030,1031,1034,1035,1036,1037,1038,1039,1040,1041,1042,1043,1044,1045,1046,1047 },
        "1002:1005,1007:1010,1013:1023,1025:1026,1028:1031,1034:1047" },

    { { 1000,1001,1002,1003,1004,1005,1006,1008,1009,1010,1012,1014,1016,1018,1019,1021,1022,1023,
        1026,1027,1028,1029,1030,1031,1033,1034,1035,1036,1038,1039,1040,1041,1042,1044,1046,1047,1048,1049 },
        "1000:1006,1008:1010,1012,1014,1016,1018:1019,1021:1023,1026:1031,1033:1036,1038:1042,1044,1046:1049" },

    { { 1000,1002,1003,1005,1006,1009,1010,1011,1015,1016,1018,1019,1020,1025,1026,1027,1029,1030,
        1033,1036,1037,1038,1042,1043,1045,1046,1047,1048,1049 },
        "1000,1002:1003,1005:1006,1009:1011,1015:1016,1018:1020,1025:1027,1029:1030,1033,1036:1038,1042:1043,1045:1049" },

    { { 1000,1001,1004,1005,1006,1012,1013,1014,1015,1016,1017,1018,1019,1022,1023,1024,1025,1026,
        1027,1028,1031,1034,1036,1037,1038,1039,1041,1045,1047,1048,1049 },
        "1000:1001,1004:1006,1012:1019,1022:1028,1031,1034,1036:1039,1041,1045,1047:1049" },

    { { 1000,1002,1003,1004,1005,1006,1007,1009,1010,1012,1015,1017,1018,1019,1020,1021,1023,1024,
        1025,1026,1028,1029,1032,1033,1034,1038,1040,1041,1042,1044,1045,1046,1047,1049 },
        "1000,1002:1007,1009:1010,1012,1015,1017:1021,1023:1026,1028:1029,1032:1034,1038,1040:1042,1044:1047,1049" },

    { { 1002,1003,1006,1008,1009,1011,1012,1014,1015,1017,1018,1019,1020,1022,1023,1024,1025,1027,
        1029,1031,1032,1033,1035,1036,1038,1039,1040,1043,1044,1045,1046,1048,1049 },
        "1002:1003,1006,1008:1009,1011:1012,1014:1015,1017:1020,1022:1025,1027,1029,1031:1033,1035:1036,1038:1040,1043:1046,1048:1049" },

    { { 1000,1001,1002,1003,1004,1006,1008,1010,1011,1012,1018,1020,1024,1025,1026,1028,1029,1030,
        1031,1032,1033,1035,1036,1038,1042,1043,1044,1045,1046,1047,1049 },
        "1000:1004,1006,1008,1010:1012,1018,1020,1024:1026,1028:1033,1035:1036,1038,1042:1047,1049" },

    { { 1000,1001,1002,1003,1004,1007,1008,1009,1011,1013,1017,1018,1019,1020,1021,1023,1024,1025,
        1026,1027,1028,1029,1030,1033,1035,1036,1037,1041,1042,1043,1045,1046,1047,1048,1049 },
        "1000:1004,1007:1009,1011,1013,1017:1021,1023:1030,1033,1035:1037,1041:1043,1045:1049" },

    { { 1000,1001,1002,1003,1004,1005,1007,1008,1009,1011,1012,1014,1015,1017,1018,1019,1021,1023,
        1024,1025,1026,1027,1028,1029,1030,1031,1032,1035,1037,1038,1040,1042,1043,1047,1048 },
        "1000:1005,1007:1009,1011:1012,1014:1015,1017:1019,1021,1023:1032,1035,1037:1038,1040,1042:1043,1047:1048" },

    { { 1000,1001,1002,1005,1008,1009,1010,1011,1012,1013,1015,1016,1018,1019,1020,1021,1022,1023,
        1024,1025,1027,1028,1029,1030,1031,1032,1035,1036,1037,1038,1039,1041,1042,1044,1045,1047,1048,1049 },
        "1000:1002,1005,1008:1013,1015:1016,1018:1025,1027:1032,1035:1039,1041:1042,1044:1045,1047:1049" },
    // clang-format on
  };

  for (int i = 0; i < mutt_array_size(tests); i++)
  {
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    const struct TestCase *test = &tests[i];
    for (int j = 0; test->input[j] > 0; j++)
    {
      ARRAY_ADD(&uida, test->input[j]);
    }

    int pos = 0;
    struct Buffer *msg_set = buf_pool_get();
    int rc = imap_make_msg_set(&uida, msg_set, &pos);

    TEST_CHECK(rc == ARRAY_SIZE(&uida));
    TEST_CHECK_STR_EQ(buf_string(msg_set), test->result);

    buf_pool_release(&msg_set);
    ARRAY_FREE(&uida);
  }
}

void test_exec(void)
{
  TEST_CASE("exec");

  static const struct TestCase tests[] = {
    // clang-format off
    { { 1, 2, 3, 7, 8, 9, 20, 21, 22 },
        "PRE 1:3,7:9,20:22 POST\n" },

    { { 1000,1003,1004,1005,1006,1007,1008,1009,1011,1012,1013,1014,1015,1016,1017,1018,1019,1021,1022,1023,1024,1025 },
        "PRE 1000,1003:1009,1011:1019,1021:1025 POST\n" },

    { { 1000,1003,1004,1005,1006,1007,1008,1009,1011,1012,1013,1014,1015,1016,1017,1018,1019,1021,
        1022,1023,1024,1025,1027,1028,1029,1030,1032,1033,1034,1036,1037,1038,1039,1040,1042,1044,1046,1049 },
        "PRE 1000,1003:1009,1011:1019,1021:1025,1027:1030,1032:1034,1036 POST\n"
        "PRE 1037:1040,1042,1044,1046,1049 POST\n" },

    { { 1000,1001,1003,1004,1005,1007,1008,1009,1010,1011,1012,1013,1014,1015,1016,1017,1018,1020,1023,
        1025,1026,1027,1028,1029,1030,1031,1032,1033,1034,1035,1036,1037,1038,1039,1040,1042,1043,1044,
        1045,1046,1047,1048,1049,1050,1052,1053,1054,1056,1057,1060,1061,1062,1064,1065,1068,1069,1070,
        1074,1075,1078,1079,1080,1081,1082,1083,1085,1087,1088,1089,1090,1092,1094,1095,1096,1098,1099,
        1101,1102,1103,1104,1105,1106,1107,1108,1109,1110,1111,1112,1113,1114,1115,1119,1120,1121,1122,
        1124,1125,1127,1128,1130,1131,1133,1134,1135,1136,1138,1140,1141,1142,1144,1147,1149 },
        "PRE 1000:1001,1003:1005,1007:1018,1020,1023,1025:1040,1042 POST\n"
        "PRE 1043:1050,1052:1054,1056:1057,1060:1062,1064:1065,1068 POST\n"
        "PRE 1069:1070,1074:1075,1078:1083,1085,1087:1090,1092,1094 POST\n"
        "PRE 1095:1096,1098:1099,1101:1115,1119:1122,1124:1125,1127 POST\n"
        "PRE 1128,1130:1131,1133:1136,1138,1140:1142,1144,1147,1149 POST\n" },
    // clang-format on
  };

  {
    imap_exec_fail = true;
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    ARRAY_ADD(&uida, 1);

    int rc = imap_exec_msg_set(NULL, "PRE", "POST", &uida);
    TEST_CHECK(rc == -1);

    ARRAY_FREE(&uida);
    imap_exec_fail = false;
  }

  // Clamp down the limit to force wrapping
  ImapMaxCmdlen = 50;
  imap_exec_results = buf_pool_get();

  for (int i = 0; i < mutt_array_size(tests); i++)
  {
    struct UidArray uida = ARRAY_HEAD_INITIALIZER;
    const struct TestCase *test = &tests[i];
    for (int j = 0; test->input[j] > 0; j++)
    {
      ARRAY_ADD(&uida, test->input[j]);
    }

    buf_reset(imap_exec_results);

    int rc = imap_exec_msg_set(NULL, "PRE", "POST", &uida);
    TEST_CHECK(rc > 0);
    TEST_CHECK_STR_EQ(buf_string(imap_exec_results), test->result);

    ARRAY_FREE(&uida);
  }

  buf_pool_release(&imap_exec_results);
}

void test_imap_msg_set(void)
{
  test_sort();
  test_make_degenerate();
  test_make_simple();
  test_make_curated();
  test_exec();
}
