<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4 autoindent: */
/**
 * Numbers_Words class extension to spell numbers in British English.
 *
 * PHP versions 4 and 5
 *
 * LICENSE: This source file is subject to version 3.01 of the PHP license
 * that is available through the world-wide-web at the following URI:
 * http://www.php.net/license/3_01.txt.  If you did not receive a copy of
 * the PHP License and are unable to obtain it through the web, please
 * send a note to license@php.net so we can mail you a copy immediately.
 *
 * @category  Numbers
 * @package   Numbers_Words
 * @author    Mtys Somfai <somfai.matyas@gmail.com>
 * @license   http://www.php.net/license/3_01.txt  PHP License 3.01
 * @link      http://pear.php.net/package/Numbers_Words
 */

/*
 * PEAR QA suggests that we do not set
 * error_reporting
 *
 * error_reporting(E_ALL | E_STRICT);
 */
ini_set('display_errors', 1);

if (!defined('PHPUnit_MAIN_METHOD')) {
    define('PHPUnit_MAIN_METHOD', 'Numbers_Words_HungarianTest::main');
}

require_once 'Numbers/Words.php';
require_once 'PHPUnit/Framework/TestCase.php';

class Numbers_Words_HungarianTest extends PHPUnit_Framework_TestCase
{
    var $handle;
    var $lang = 'hu_HU';

    public static function main()
    {
        require_once 'PHPUnit/TextUI/TestRunner.php';
        PHPUnit_TextUI_TestRunner::run(
            new PHPUnit_Framework_TestSuite('Numbers_Words_HungarianTest')
        );
    }

    function setUp()
    {
        $this->handle = new Numbers_Words();
    }

    /**
     * Testing numbers between 0 and 9
     */
    function testDigits()
    {
        $digits = array('nulla',
                        'egy',
                        'kett',
                        'hrom',
                        'ngy',
                        't',
                        'hat',
                        'ht',
                        'nyolc',
                        'kilenc'
                       );
        for ($i = 0; $i < 10; $i++) {
            $number = $this->handle->toWords($i, $this->lang);
            $this->assertEquals($digits[$i], $number);
        }
    }

    /**
     * Testing numbers between 10 and 99
     */
    function testTens()
    {
        $tens = array(11 => 'tizenegy',
                      12 => 'tizenkett',
                      16 => 'tizenhat',
                      19 => 'tizenkilenc',
                      20 => 'hsz',
                      21 => 'hszonegy',
                      26 => 'hszonhat',
                      30 => 'harminc',
                      31 => 'harmincegy',
                      40 => 'negyven',
                      43 => 'negyvenhrom',
                      50 => 'tven',
                      55 => 'tvent',
                      60 => 'hatvan',
                      67 => 'hatvanht',
                      70 => 'hetven',
                      79 => 'hetvenkilenc'
                     );
        foreach ($tens as $number => $word) {
            $this->assertEquals($word, $this->handle->toWords($number, $this->lang));
        }
    }

    /**
     * Testing numbers between 100 and 999
     */
    function testHundreds()
    {
        $hundreds = array(100 => 'egyszz',
                          101 => 'egyszzegy',
                          199 => 'egyszzkilencvenkilenc',
                          203 => 'kettszzhrom',
                          287 => 'kettszznyolcvanht',
                          300 => 'hromszz',
                          356 => 'hromszztvenhat',
                          410 => 'ngyszztz',
                          434 => 'ngyszzharmincngy',
                          578 => 'tszzhetvennyolc',
                          689 => 'hatszznyolcvankilenc',
                          729 => 'htszzhszonkilenc',
                          894 => 'nyolcszzkilencvenngy',
                          999 => 'kilencszzkilencvenkilenc'
                         );
        foreach ($hundreds as $number => $word) {
            $this->assertEquals($word, $this->handle->toWords($number, $this->lang));
        }
    }

    /**
     * Testing numbers between 1000 and 9999
     */
    function testThousands()
    {
        $thousands = array(1000 => 'egyezer',
                           1001 => 'egyezeregy',
                           1097 => 'egyezerkilencvenht',
                           1104 => 'egyezeregyszzngy',
                           1243 => 'egyezerkettszznegyvenhrom',
                           2385 => 'kettezer-hromszznyolcvant',
                           3766 => 'hromezer-htszzhatvanhat',
                           4196 => 'ngyezer-egyszzkilencvenhat',
                           5846 => 'tezer-nyolcszznegyvenhat',
                           6459 => 'hatezer-ngyszztvenkilenc',
                           7232 => 'htezer-kettszzharminckett',
                           8569 => 'nyolcezer-tszzhatvankilenc',
                           9539 => 'kilencezer-tszzharminckilenc'
                          );
        foreach ($thousands as $number => $word) {
            $this->assertEquals($word, $this->handle->toWords($number, $this->lang));
        }
    }
    
    /**
    */
    function testMore()
    {

        $this->assertEquals('egymilli', $this->handle->toWords(1000000, $this->lang));
		$this->assertEquals('egymilli-egyezer-tszz', $this->handle->toWords(1001500, $this->lang));
		$this->assertEquals('kettmilli-egy', $this->handle->toWords(2000001, $this->lang));
		$this->assertEquals('nyolcmilli-kettezer-egy', $this->handle->toWords(8002001, $this->lang));
        $this->assertEquals('kettmillird', $this->handle->toWords(2000000000, $this->lang));


        // 32 bit systems vs PHP_INT_SIZE - 3 billion is a little high, so use a string version.
        $number = '3000000000000' > PHP_INT_SIZE? '3000000000000' : 3000000000000;

        $this->assertEquals('hrombilli', $this->handle->toWords($number, $this->lang));
    
    }
}

if (PHPUnit_MAIN_METHOD == 'Numbers_Words_HungarianTest::main') {
    Numbers_Words_HungarianTest::main();
}
