/*
 * conf.c  Read the configuration file.
 *
 * Version:  @(#)conf.c  1.25  08-Nov-1997  MvS.
 *
 */
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stddef.h>

#define EXTERN
#include "server.h"

#define C_INT    1
#define C_STR    2
#define C_HOST   3
#define C_IPNO   4
#define C_IPDY   5
#define C_IP6NO  6
#define C_CHAT   7
#define C_REALM  8
#define C_INDY   9
#define C_BOOL  10
#define C_HSRV  11
#define C_DEV   12
#define C_LIST  13
#ifdef CYCLADES
#define C_XSTR  200
#define C_XIPNO 201
#define C_XINT  202
#endif

typedef enum
{
  parseOK = 0,
  parseErr = -1,
  parseFatal = -2
} PARSE_RC;

/*
 *  For option lists.
 */
struct lst
{
  const char *name;
  int value;
};

/*
 *  Define type of config variables.
 */
struct conf
{
  const char *name;
  int type;
  const struct lst *opts;
  void *ptr;
  bool really_set;
};

/*
 *  Types of authentication.
 */
const struct lst radlst[] = {
  { "none",         AUTH_NONE         },
  { "radius",       AUTH_RADIUS       },
  { "tacacs",       AUTH_TACACS       },
  { "remote",       AUTH_REMOTE       },
  { "local",        AUTH_LOCAL        },
  { "radius/local", AUTH_RADIUS_LOCAL },
  { "tacacs/local", AUTH_TACACS_LOCAL },
  { "local/radius", AUTH_LOCAL_RADIUS },
  { "local/tacacs", AUTH_LOCAL_TACACS },
#ifdef CYCLADES
  { "RadiusDownLocal", AUTH_RADIUS_DOWN_LOCAL  },
  { "TacacsDownLocal", AUTH_TACACS_DOWN_LOCAL  },
#endif
  { NULL,    0  }
};

/*
 *  Types of flow control.
 */
const struct lst flowlst[] = {
  { "none",  FLOW_NONE },
  { "hard",  FLOW_HARD },
  { "soft",  FLOW_SOFT },
  { NULL,  0    }
};

/*
 *     Types of parity.
 */
const struct lst paritylst[] = {
  { "none",       1 },
  { "odd",        2 },
  { "even",       3 },
  { NULL, 0}
};

/*
 *  Protocols.
 */
const struct lst prlst[] = {
  { "login",     P_LOCAL    },
  { "rlogin",    P_RLOGIN   },
  { "telnet",    P_TELNET   },
  { "ssh1",      P_SSH1      },
  { "ssh2",      P_SSH2     },
  { "ssh",       P_SSH2     },
  { "slip",      P_SLIP     },
  { "cslip",     P_CSLIP    },
  { "ppp",       P_PPP      },
  { "ppp_only",  P_PPP_ONLY },
  { "tcpclear",  P_TCPCLEAR },
  { "tcplogin",  P_TCPLOGIN },
  { "console",   P_CONSOLE  },
  { "socket_client", P_SOCKET_CLIENT },
  { "socket_server", P_SOCKET_SERVER },
  { "socket_ssh",    P_SOCKET_SSH    },
  { NULL,    0    }
};

const struct lst ptypelst[] = {
  { "0",         0 },
  { "async",     0 },
  { "1",         1 },
  { "sync",      1 },
  { "2",         2 },
  { "isdn",      2 },
  { "3",         3 },
  { "isdn-v120", 3 },
  { "4",         4 },
  { "isdn-v110", 4 },
  { NULL,        0 }
};

const struct lst daylst[] = {
  { "sun",	1 },
  { "mon",	2 },
  { "tue",	4 },
  { "wed",	8 },
  { "thu",	16 },
  { "fri",	32 },
  { "sat",	64 },
  { "week",	62 },
  { "weekend",	65 },
  { "all",	127 }
};

static PARSE_RC setlist(const char *val, const struct lst *lst, void *ptr);

/* set a single time range (set of days and hhmm-hhmm time range) */
static PARSE_RC a_login_time(struct time_ent *t_ent, char *str)
{
  int tmp_ind;
  unsigned sh, sm, eh, em;
  PARSE_RC rc;
  char *tmp_str = strtok(str, " ");

  while(tmp_str && isalpha(tmp_str[0]))
  {
    rc = setlist(tmp_str, daylst, &tmp_ind);
    if(rc)
      return rc;
    t_ent->days |= tmp_ind;
    tmp_str = strtok(NULL, " ");
  }
  if(!tmp_str)
    return parseErr;
  if(sscanf(tmp_str, "%2u%2u-%2u%2u", &sh, &sm, &eh, &em) != 4)
    return parseErr;
  if(sh > 23 || eh > 23 || sm > 59 || em > 59)
    return parseErr;
  t_ent->start_time = sh * 60 + sm;
  t_ent->end_time = eh * 60 + em;
  if(t_ent->start_time >= t_ent->end_time || t_ent->end_time > 24 * 60)
    return parseErr;
  return parseOK;
}

static void set_login_times(const char * const time_str)
{
  int        i = 0;
  int        ar_size = 2;
  char *time_range;
  char *time_tmp_str;

  while (time_str[i])
    if (time_str[i++] == ',')
      ar_size++;

  lineconf.login_time = (struct time_ent *)xmalloc(ar_size * sizeof(struct time_ent));

  time_tmp_str = xstrdup(time_str);
  time_range = strtok(time_tmp_str, ",");
  i = 0;
  while(time_range)
  {
    PARSE_RC rc;
    rc = a_login_time(&lineconf.login_time[i], time_range);
    if(rc)
    {
      free(lineconf.login_time);
      lineconf.login_time = NULL;
      free(time_tmp_str);
      nsyslog(LOG_ERR, "Can't parse time string \"%s\".", time_str);
      return;
    }
    i++;
    time_range = strtok(NULL, ",");
  }
  lineconf.login_time[i].days = 0;
  free(time_tmp_str);
}

/*
 *  Define structure for the per-line configuration.
 */
static struct conf line_cfg[] = {
{ "hostname",    C_STR,  NULL,    &lineconf.hostname, 0 },
{ "radclient_config_file",    C_STR,  NULL,    &lineconf.radclient_config_file, 0 },
{ "loc_host",    C_IPNO, NULL,    &lineconf.loc_host, 0 },
#ifdef HAVE_IPV6
{ "loc_host6",   C_IP6NO,NULL,    &lineconf.loc_host6, 0 },
{ "use_v6",      C_BOOL,  NULL,   &lineconf.use_v6, 0 },
#endif
{ "lockdir",     C_STR,  NULL,    &lineconf.lockdir, 0 },
{ "rlogin",      C_STR,  NULL,    &lineconf.rlogin, 0 },
{ "telnet",      C_STR,  NULL,    &lineconf.telnet, 0 },
{ "ssh",         C_STR,  NULL,    &lineconf.ssh, 0 },
{ "pppd",        C_STR,  NULL,    &lineconf.pppd, 0 },
{ "locallogins", C_BOOL, NULL,    &lineconf.locallogins, 0 },
{ "syslog",      C_HOST, NULL,    &lineconf.syslog, 0 },
{ "facility",    C_INT,  NULL,    &lineconf.facility, 0 },
{ "filterdir",   C_STR,  NULL,    &lineconf.filterdir, 0 },
{ "stripnames",  C_BOOL, NULL,    &lineconf.stripnames, 0 },
{ "debug",       C_INT,  NULL,    &lineconf.debug, 0 },
{ "sysutmp",     C_BOOL, NULL,    &lineconf.sysutmp, 0 },
{ "syswtmp",     C_BOOL, NULL,    &lineconf.syswtmp, 0 },
{ "utmpfrom",    C_STR,  NULL,    &lineconf.utmpfrom, 0 },
{ "emumodem",    C_BOOL, NULL,    &lineconf.emumodem, 0 },
{ "porttype",    C_LIST, ptypelst,&lineconf.porttype, 0 },
{ "authtype",    C_LIST, radlst,  &lineconf.authtype, 0 },
{ "radnullpass", C_BOOL, NULL,    &lineconf.radnullpass, 0 },
#ifdef PORTSLAVE_TACACS
{ "tacauthhost1",C_HOST, NULL,    &lineconf.tacauthhost1, 0 },
{ "tacauthhost2",C_HOST, NULL,    &lineconf.tacauthhost2, 0 },
#endif
{ "protocol",    C_LIST, prlst,   &lineconf.protocol, 0 },
{ "host",        C_HOST, NULL,    &lineconf.host, 0 },
{ "rem_host",    C_IPDY, NULL,    &lineconf.rem_host, 0 },
{ "netmask",     C_IPNO, NULL,    &lineconf.netmask, 0 },
{ "mtu",         C_INT,  NULL,    &lineconf.mtu, 0 },
{ "mru",         C_INT,  NULL,    &lineconf.mru, 0 },
{ "autoppp",     C_STR,  NULL,    &lineconf.autoppp, 0 },
{ "pppopt",      C_STR,  NULL,    &lineconf.pppopt, 0 },
{ "tty",         C_DEV,  NULL,    &lineconf.tty, 0 },
{ "issue",       C_STR,  NULL,    &lineconf.issue, 0 },
{ "prompt",      C_STR,  NULL,    &lineconf.prompt, 0 },
{ "term",        C_STR,  NULL,    &lineconf.term, 0 },
{ "speed",       C_INT,  NULL,    &lineconf.speed, 0 },
{ "socket_port", C_INDY, NULL,    &lineconf.socket_port, 0 },
{ "parity",      C_LIST, paritylst,&lineconf.parity, 0},
{ "stopbits",    C_INT,  NULL,    &lineconf.stopbits, 0},
{ "datasize",    C_INT,  NULL,    &lineconf.datasize, 0},
{ "dcd",         C_BOOL, NULL,    &lineconf.dcd, 0 },
{ "flow",        C_LIST, flowlst, &lineconf.flow, 0 },
{ "initchat",    C_CHAT, NULL,    &lineconf.initchat, 0},
#ifdef FIDO
{ "fidonet",     C_BOOL, NULL,    &lineconf.fidonet, 0},
{ "fidologin",   C_STR,  NULL,    &lineconf.fidologin, 0},
{ "fidopasswd",  C_STR,  NULL,    &lineconf.fidopasswd, 0},
#endif
#ifdef PORTSLAVE_CLIENT_IP_RULES
{ "valid_ip",    C_STR,  NULL,    &lineconf.valid_ip, 0 },
#endif
{ "logpassword", C_REALM,NULL,    &lineconf.logpassword, 0 },
{ "fixedlogin",  C_STR,  NULL,    &lineconf.fixedlogin, 0 },
{ "do_acct",     C_BOOL, NULL,    &lineconf.do_acct, 0 },
#ifndef NO_CHAP
{ "allow_chap",  C_BOOL, NULL,    &lineconf.allow_chap, 0 },
#endif
{ "login_time",  C_STR, NULL,     &lineconf.login_time_str, 0 },
{ "login_time_limited",  C_BOOL, NULL,     &lineconf.login_time_limited, 0 },
{ NULL, 0, NULL, NULL, 0 }
};

/*
 *  Set an integer.
 */
static PARSE_RC setint(const char *val, void *ptr)
{
  if(sscanf(val, "%d", (int *)ptr) != 1)
    return parseErr;
  return parseOK;
}

/*
 *  Set a dynamic integer.
 */
static PARSE_RC setintdy(const char *val, void *ptr)
{
  int n;

  if(sscanf(val, "%d", &n) != 1)
    return parseErr;
  if(val[strlen(val) - 1] == '+')
    n += GetPortNo();
  *(int *)ptr = n;

  return parseOK;
}

/* Actually copy a string to the data structures, used by setstr() and setdev()
 * We own the data pointed to by val!!!
 */
static PARSE_RC real_setstr(char *val, void *ptr)
{
  char **sptr;
  sptr =(char **)ptr;
  if(*sptr)
    free(*sptr);
  *sptr = val;
  return parseOK;
}

/*
 * Set a string.
 * do_unescape is a boolean, if set then we replace \n with a new line, \r with
 * CR, etc.
 */
static PARSE_RC setstr(const char *val, void *ptr, bool do_unescape)
{
  char *myval = xstrdup(val);

  /*
   *  Specialcase "" as the empty string.
   */
  if(do_unescape)
  {
    unescape(myval);
  }
  else if (strcmp(myval, "\"\"") == 0)
  {
    free(myval);
    myval = xstrdup("");
  }

  return real_setstr(myval, ptr);
}

/* Set a device name */
static PARSE_RC setdev(const char *val, void *ptr)
{
  char *dev_name;
  if(strcmp(val, "\"\"") == 0)
    return parseErr;

  dev_name = check_device(val);
  if(!dev_name)
  {
    nsyslog(LOG_ERR, "Device \"%s\" doesn't exist.", val);
    return parseFatal;
  }
  return real_setstr(dev_name, ptr);
}

/*
 *  Set a hostname. Return ipno.
 */
static PARSE_RC sethost(const char *val, void *ptr)
{
  unsigned int n;
  struct hostent *h;

  if (val[0] == 0)
  {
    n = 0;
  }
  else
  {
    if ((int)(n = inet_addr(val)) == -1)
    {
      if((h = gethostbyname(val)) == NULL)
      {
        nsyslog(LOG_ERR, "Can't resolve host: %s", val);
        return parseFatal;
      }
      n = *(unsigned int *)h->h_addr_list[0];
    }
  }

  *(unsigned int *)ptr = n;

  return parseOK;
}

/*
 *  Set a hostname. Return ipno.
 *  Modifies val.
 */
static PARSE_RC sethost_srv(char *val, void *ptr)
{
  char *tmp = NULL;
  struct sockaddr **sptr_ptr = (struct sockaddr **)ptr;
  struct sockaddr_in *sptr = NULL;
#ifdef HAVE_IPV6
  struct sockaddr_in6 *sptr6 = NULL;
#endif

  if(*val == '[')
  {
    val++;
    tmp = strchr(val, ']');
    if(!tmp)
      return parseErr;
    *tmp = '\0';
    tmp++;
  }
  if(*sptr_ptr)
    free(*sptr_ptr);
  if( (val[0] < '0' || val[0] > '9')
#ifdef HAVE_IPV6
    && !strchr(val, ':')
#endif
    )
  {
    struct hostent *h;
#ifdef HAVE_IPV6
    h = gethostbyname2(val, AF_INET6);
    if(h)
    {
      *sptr_ptr = xmalloc(sizeof(struct sockaddr_in6));
      sptr6 = (struct sockaddr_in6*)*sptr_ptr;
      sptr6->sin6_family = AF_INET6;
      memcpy(&sptr6->sin6_addr, h->h_addr_list[0], sizeof(sptr6->sin6_addr) );
    }
    else
#endif
    {
      h = gethostbyname2(val, AF_INET);
      if(!h)
        return parseErr;
      *sptr_ptr = xmalloc(sizeof(struct sockaddr_in));
      sptr = (struct sockaddr_in*)*sptr_ptr;
      sptr->sin_family = AF_INET;
      memcpy(&sptr->sin_addr, h->h_addr_list[0], sizeof(sptr->sin_addr) );
    }
  }
  else
  {
#ifdef HAVE_IPV6
    if(strchr(val, ':'))
    {
      *sptr_ptr = xmalloc(sizeof(struct sockaddr_in6));
      sptr6 = (struct sockaddr_in6*)*sptr_ptr;
      sptr6->sin6_family = AF_INET6;
      if(inet_pton(AF_INET6, val, &sptr6->sin6_addr) <= 0)
        return parseErr;
    }
    else
#endif
    {
      *sptr_ptr = xmalloc(sizeof(struct sockaddr_in));
      sptr = (struct sockaddr_in*)*sptr_ptr;
      sptr->sin_family = AF_INET;
      if(inet_pton(AF_INET, val, &sptr->sin_addr) <= 0)
        return parseErr;
    }
  }
  if(tmp)
  {
    unsigned short port_num;
    if(isdigit(tmp[0]))
    {
      port_num = htons(atoi(tmp));
    }
    else
    {
      struct servent *svp = getservbyname(tmp, "udp");
      if(!svp)
        return parseErr;
      port_num = svp->s_port;
    }
#ifdef HAVE_IPV6
    if(sptr6)
      sptr6->sin6_port = port_num;
    else
#endif
      sptr->sin_port = port_num;
  }
  return parseOK;
}

/*
 *  Set an IP address.
 */
static PARSE_RC setipno(const char *val, void *ptr)
{
  unsigned int n;

  if((int)(n = inet_addr(val)) == -1 && strcmp(val, "255.255.255.255") != 0)
    return parseErr;

  *(unsigned int *)ptr = n;

  return parseOK;
}

/*
 *  Set a dynamic IP address.
 *  If the IP address ends in '+' then ip address is base + line num.
 */
static PARSE_RC setipdy(const char *val, void *ptr)
{
  int len;
  unsigned int base_h, base;
  int dy = 0;
  int port;

  char *myval = xstrdup(val);

  port = GetPortNo();

  len = strlen(myval);
  if(len > 0 && myval[len - 1] == '+')
  {
    dy = 1;
    myval[len - 1] = 0;
  }

  if((base = inet_addr(myval)) == INADDR_NONE && strcmp(myval, "255.255.255.255") != 0)
  {
    free(myval);
    return parseErr;
  }
  base_h = ntohl(base); /* base_h is base IP in host byte order */

  if(port >= 0 && dy)
    base = htonl(base_h + port);
  *(unsigned int *)ptr = base;

  free(myval);
  return parseOK;
}

/*
 *  Set an IPv6 address.
 */
static PARSE_RC setip6no(const char *val, void *ptr)
{
  if(inet_pton(AF_INET6, val, ptr) <= 0)
    return parseErr;

  return parseOK;
}

/*
 *  Set an integer, from a list of string values.
 */
static PARSE_RC setlist(const char *val, const struct lst *lst, void *ptr)
{
  int n = 0, i;

  for(i = 0; lst[i].name; i++)
  {
    if(strcasecmp(val, lst[i].name) == 0)
    {
      n = lst[i].value;
      break;
    }
  }
  if(lst[i].name == NULL) return parseErr;

  *(int *)ptr = n;

  return parseOK;
}

static PARSE_RC setbool(const char *val, void *ptr)
{
  if(!strcmp("0", val) || !strcmp("false", val) || !strcmp("no", val))
    *(bool *)ptr = false;
  else if(!strcmp("1", val) || !strcmp("true", val) || !strcmp("yes", val))
    *(bool *)ptr = true;
  else
    return parseErr;
  return parseOK;
}

/* if the word is "s1" and our port number is 1 then return 0.
 * if the word if "s{5-9} and our port number is s7 then return 2 (the index
 * into the list).
 * If the word is for another port then return -1.
 * If a parse error then return -2;
 *
 */
static int isLineForOurPort(const char *word, int *list_size)
{
  int start, end;

  if(tolower(*word) != 's')
    return -2;
  word++;
  if(*word == '{')
  {
    if(2 != sscanf(word + 1, "%d-%d", &start, &end) || start >= end)
      return -2;
  }
  else
  {
    if( !(start = atoi(word)) && *word != '0')
      return -2;
    *list_size = 0;
    if(GetPortNo() == start)
      return 0;
    return -1;
  }
  if(start <= GetPortNo() && GetPortNo() <= end)
  {
    *list_size = end - start;
    return GetPortNo() - start;
  }
  return -1;
}

/* replaces a {start-end} string in the arguement with the list_index number.
 * writes over the same string data as the result will be shorter and the
 * caller is only going to strdup() it and free it afterwards.
 */
static int convert_argument(char *arg, int list_index, int list_size)
{
  int start, end;
  char *argument, *end_ptr;

  argument = strchr(arg, '{');
  end_ptr = strchr(arg, '}');
  if(!argument && !end_ptr)
    return parseOK;
  if(!argument || !end_ptr)
    return parseErr;
  end_ptr++;
  if(2 != sscanf(argument + 1, "%d-%d", &start, &end) || start >= end)
    return parseErr;
  if(list_size != (end - start) || end <= start )
    return parseErr;
  sprintf(argument, "%d", start + list_index);
  argument += strlen(argument);
  if(argument < end_ptr + 1 && *end_ptr != 0)
    memmove(argument, end_ptr, strlen(end_ptr) + 1);
  return parseOK;
}

/*
 *  Parse one line from the configuration file.
 */
static PARSE_RC parseline(const char *line)
{
  char *dup_line = xstrdup(line);
  char *s = dup_line;
  bool all_entry = false;
  char *word, *argument, *command;
  struct conf *x;
  int n = parseErr;
  int list_index = 0, list_size = 0;

  /*
   *  Remove the trailing newline and spaces etc.
   */
  for(word = s + strlen(s) - 1; isspace(*word) || *word == '\n' || *word == '\r'; word--)
  {
    *word = '\0';
    if(word <= s)
      break;
  }

  /*
   *  Get word, command and argument.
   */
  while(isspace(*s)) s++;
  word = s;
  while(*s && !isspace(*s)) s++;
  if(*word == 0)
  {
parseline_error:
    free(dup_line);
    return n;
  }
  if (*s) *s++ = 0;
  while(isspace(*s)) s++;
  argument = s;
  if((command = strchr(word, '.')) == NULL || command[1] == 0)
    goto parseline_error;
  *command++ = 0;

  /*
   *  See what class of command this is.
   */
  if(!strcasecmp(word, "conf") || !strcasecmp(word, "all"))
  {
    all_entry = true;
  }
  else
  {
    list_index = isLineForOurPort(word, &list_size);
    switch(list_index)
    {
    case -2:
      free(dup_line);
      return parseErr;
    case -1:
      free(dup_line);
      return parseOK;
    break;
    }
  }

  /*
   *  Parse the line and fill in the right structure(s).
   */
  for(x = line_cfg; x->name != NULL; x++)
  {
    if(strcasecmp(command, x->name) != 0) continue;
/* if we are already set and this is an all entry then return OK */
    if(x->really_set && all_entry)
    {
      free(dup_line);
      return parseOK;
    }
    if(x->type != C_STR && x->type != C_CHAT)
    {
      n = convert_argument(argument, list_index, list_size);
      if(n != parseOK)
        goto parseline_error;
    }
    x->really_set = 1;
    switch(x->type)
    {
      case C_INDY:
        n = setintdy(argument, x->ptr);
      break;
      case C_INT:
        n = setint(argument, x->ptr);
      break;
      case C_BOOL:
        n = setbool(argument, x->ptr);
      break;
      case C_STR:
        n = setstr(argument, x->ptr, true);
      break;
      case C_DEV:
        n = setdev(argument, x->ptr);
      break;
      case C_CHAT:
        n = setstr(argument, x->ptr, false);
      break;
      case C_HOST:
        n = sethost(argument, x->ptr);
      break;
      case C_HSRV:
        n = sethost_srv(argument, x->ptr);
      break;
      case C_LIST:
        n = setlist(argument, x->opts, x->ptr);
      break;
      case C_IPNO:
        n = setipno(argument, x->ptr);
      break;
      case C_IP6NO:
        n = setip6no(argument, x->ptr);
      break;
      case C_IPDY:
        n = setipdy(argument, x->ptr);
      break;
      default:
        n = parseFatal;
      break;
    }
    break;
  }
  if(x->name == NULL || n != parseOK) goto parseline_error;

  free(dup_line);

  return parseOK;
}

/*
 *  Check if the line specifies the port we use
 */
static int check_portno(const char *buf, const char *tty)
{
  int port;
  int rc = 0;
  char *dev;
  int start, end, list_size = 0;
  char *argument, *end_ptr;
  int i;

  if(buf[0] != 's')
    return -1;
  buf++;
  if(buf[0] == '{')
  {
    end_ptr = strstr(buf, "}.");
    if(!end_ptr)
      return -1;
    buf++;
    if(2 != sscanf(buf, "%d-%d", &port, &end) || port >= end)
      return -1;
    list_size = end - port;
  }
  else
  {
    if(strchr(buf, '{') || strchr(buf, '}') )
      return -1;
    port = atoi(buf);
    if(port < 0)
      return -1;
  }
  buf = strchr(buf, '.');
  if(!buf)
    return -1;
  if(!strncmp(buf, ".tty", 4))
    buf += 4;
  else
    return -1;
  while(isspace(*buf))
    buf++;

  if(list_size != 0)
  {
    argument = strchr(buf, '{');
    end_ptr = strchr(buf, '}');
    if(!argument || !end_ptr)
      return -1;
    end_ptr++;
    if(2 != sscanf(argument + 1, "%d-%d", &start, &end) || start >= end
      || list_size != (end - start) )
      return -1;
    end_ptr = xstrdup(end_ptr);
    for(i = start; i <= end; i++)
    {
      sprintf(argument, "%d%s", start + i, end_ptr);
      dev = check_device(buf);
      if(dev && !strcmp(dev, tty))
      {
        SetPortNo(port + i);
        rc = 0;
        free(dev);
        return 0;
      }
    }
    free(end_ptr);
    return -1;
  }

  dev = check_device(buf);
  if(!dev)
    return -1;
  if(strcmp(dev, tty))
    rc = -1;
  else
    SetPortNo(port);
  free(dev);
  return rc;
}

/*
 *  Read the configuration file.
 */
int readcfg(const char *config_file, const char *tty)
{
  FILE *fp;
  char buf[2048];
  int lineno = 0;
  char *s, *p;

  if(!tty)
    tty = ttyname(0);
  lineconf.tty = xstrdup(tty);
  if((fp = fopen(config_file, "r")) == NULL)
  {
    nsyslog(LOG_ERR, "%s: %m", config_file);
    return -1;
  }
  p = buf;
  if(GetPortNo() == RAD_INIT_PPPD)
  {
    const char *env_data = getenv("PORTSLAVE_PORT");
    if(env_data)
      SetPortNo(atoi(env_data));
  }
  if(GetPortNo() == RAD_INIT_PPPD)
  {
    while(fgets(p, sizeof(buf) - (p - buf), fp))
    {
      if (p[0] == '#' || p[0] == '\n') continue;
      s = p + strlen(p);
      if (s > p && *--s == '\n' && *--s == '\\')
      {
        p = s;
        continue;
      }
      if (buf[0] == '\n' || buf[0] == 0)
      {
        p = buf;
        continue;
      }
      strtok(buf, "\r\n");
      p = buf + strlen(buf) - 1;
      while(isspace(*p))
      {
        *p = '\0';
        p--;
      }
      if(!check_portno(buf, tty) )
        break;
      p = buf;
    }
    if(GetPortNo() == RAD_INIT_PPPD)
    {
      fclose(fp);
      nsyslog(LOG_ERR, "Port %s not defined in %s.\n"
        , lineconf.tty, config_file);
      return -1;
    }

    /* reopen the log after setting the port */
    nopenlog(NULL, LOG_NDELAY, 0);
    if(fseek(fp, 0, SEEK_SET))
    {
      fclose(fp);
      nsyslog(LOG_ERR, "%s: seek error\n", config_file);
      return -1;
    }
  }
  p = buf;
  while(fgets(p, sizeof(buf) - (p - buf), fp))
  {
    int rc;
    lineno++;
    if (p[0] == '#' || p[0] == '\n') continue;
    s = p + strlen(p);
    if (s > p && *--s == '\n' && *--s == '\\')
    {
      p = s;
      continue;
    }
    if (buf[0] == '\n' || buf[0] == 0)
    {
      p = buf;
      continue;
    }
    rc = parseline(buf);
    if(rc == parseFatal)
    {
      nsyslog(LOG_ERR, "%s[%d]: fatal parse error\n", config_file, lineno);
      fclose(fp);
      return -1;
    }
    if(rc < 0)
      nsyslog(LOG_WARNING, "%s[%d]: syntax error\n", config_file, lineno);
    p = buf;
  }
  fclose(fp);
  /* reopen the log after getting all the config data */
  nopenlog(NULL, LOG_NDELAY, 0);
  if(lineconf.fixedlogin && !strlen(lineconf.fixedlogin))
  {
    free(lineconf.fixedlogin);
    lineconf.fixedlogin = NULL;
  }
  if(lineconf.login_time_str)
  {
    set_login_times(lineconf.login_time_str);
    free(lineconf.login_time_str);
    lineconf.login_time_str = NULL;
  }
  return 0;
}

#define REPLACE_STR(Xdest, Xsrc) { if(Xdest) free(Xdest); Xdest = Xsrc; }

/*
 *  Initialize the configuration stuff with defaults.
 */
void initcfg(void)
{
  struct hostent *h;
  char buf[256];
  char telnet [] = PATH_TELNET;
  char ssh [] = PATH_SSH;
  char rlogin [] = PATH_RLOGIN;
  char pppd [] = PATH_PPP_RADIUS;
  
  gethostname(buf, sizeof(buf));
  if(lineconf.hostname)
    free(lineconf.hostname);
  lineconf.hostname = xstrdup(buf);
  if(( h = gethostbyname(buf)) != NULL)
    lineconf.loc_host = *(unsigned int *)(h->h_addr);

  REPLACE_STR(lineconf.prompt, xstrdup("%h login: "));
  lineconf.radnullpass = 1;
  REPLACE_STR(lineconf.utmpfrom, xstrdup("%p:%P.%3.%4"));
  lineconf.sysutmp = 1;
  lineconf.syswtmp = 1;
  setlist("ppp", prlst, &lineconf.protocol);
  if(strlen(telnet))
  {
    REPLACE_STR(lineconf.telnet, xstrdup(telnet));
  }
  if(strlen(ssh))
  {
    REPLACE_STR(lineconf.ssh, xstrdup(ssh));
  }
  if(strlen(rlogin))
  {
    REPLACE_STR(lineconf.rlogin, xstrdup(rlogin));
  }
  REPLACE_STR(lineconf.pppd, xstrdup(pppd));
  REPLACE_STR(lineconf.lockdir, xstrdup("/var/lock"));
  REPLACE_STR(lineconf.term, xstrdup("vt100"));
  REPLACE_STR(lineconf.radclient_config_file, xstrdup(RADCLIENT_CFG));
  lineconf.stripnames = 1;
  lineconf.debug = 0;
  setlist("none", paritylst, &lineconf.parity);
  lineconf.stopbits = 1;
  lineconf.datasize = 8;

  SetChatTimeout(10);
  SetChatSendDelay(1);
}
