/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QuteCom.h"

#include <model/profile/UserProfile.h>
#include <model/profile/UserProfileHandler.h>

#include "commandserver/CommandServer.h"
#include "config/Config.h"
#include "config/ConfigManager.h"
#include "config/ConfigManagerFileStorage.h"
#include "dtmf/DtmfThemeManager.h"
#include "network/NetworkObserver.h"
#include "network/NetworkProxyDiscovery.h"

#include "QuteComBuildId.h"

#include <cutil/global.h>
#include <http/HttpRequest.h>
#include <thread/Timer.h>
#include <util/Logger.h>
#include <util/SafeDelete.h>
#include <webcam/WebcamDriver.h>

#include <sstream>

QuteCom::QuteCom() {

	_enableSerialization = true;

	//set HttpRequest User Agent
	std::stringstream ss;
	ss << QuteComBuildId::getSoftphoneName();
	ss << "-";
	ss << QuteComBuildId::getVersion();
	ss << "-";
	ss << QuteComBuildId::getBuildId();
	ss << "-";
	ss << QuteComBuildId::getSvnRevision();
	HttpRequest::setUserAgent(ss.str());
	////

	//Creating instance of NetworkObserver
	NetworkObserver::getInstance();
	////

	//Creating instance of NetworkProxyDiscovery
	NetworkProxyDiscovery::getInstance();
	////

	//Initialize libwebcam for the model thread.
	WebcamDriver::apiInitialize();
	////

	Config & config = ConfigManager::getInstance().getCurrentConfig();

	//Binding events
	config.valueChangedEvent += boost::bind(&QuteCom::valueChangedEventHandler, this, _1);
	////

	//Creating the UserProfileHandler instance
	_userProfileHandler = new UserProfileHandler();
	////

	//Creating instance of CommandServer
	CommandServer::getInstance(*this);
	////
}

void QuteCom::exitAfterTimeout() {
	exitEvent(*this);
}

QuteCom::~QuteCom() {
        // make sure QuteCom::run() has terminated
        Thread::terminate(); // should not be needed, but let's avoid potential deadlock
        Thread::join();

	//Deleting created objects
	OWSAFE_DELETE(_userProfileHandler);
	OWSAFE_DELETE(_dtmfThemeManager);
	////
	saveConfiguration();
}

void QuteCom::init() {
	Config & config = ConfigManager::getInstance().getCurrentConfig();

	//DTMFThemeManager
	_dtmfThemeManager = new DtmfThemeManager(*this, config.getAudioSmileysDir());
	//Sends the DtmfThemeManager creation event
	dtmfThemeManagerCreatedEvent(*this, *_dtmfThemeManager);

	_userProfileHandler->init();

	//initFinishedEvent
	initFinishedEvent(*this);
}

void QuteCom::run() {
	init();

	LOG_DEBUG("model thread is ready for events");

	runEvents();

	_userProfileHandler->uninit();
}

void QuteCom::terminate() {
	Thread::terminate();
}

void QuteCom::prepareToTerminate() {
	/**
	 * Set up a timeout triggered if SIP registering is too long
	 * so that closing QuteCom is not too long.
	 */
	static Timer shutdownTimeout;
	shutdownTimeout.timeoutEvent += boost::bind(&QuteCom::exitAfterTimeout, this);
	shutdownTimeout.start(3000, 3000);
}

void QuteCom::saveConfiguration() {

	if (_enableSerialization) {
		ConfigManagerFileStorage configManagerStorage(ConfigManager::getInstance());
		configManagerStorage.save();
	}
}

void QuteCom::valueChangedEventHandler(const std::string & key) {
	saveConfiguration();
}

CoIpManager * QuteCom::getCoIpManager() const {
	CoIpManager *result = NULL;
	UserProfile *userProfile = _userProfileHandler->getCurrentUserProfile();

	if (userProfile) {
		result = &userProfile->getCoIpManager();
	}

	return result;
}

void QuteCom::enableSerialization(bool enable) {
	_enableSerialization = enable;
	_userProfileHandler->enableAutoSave(enable);
}

void QuteCom::setStartupCall(const std::string& call) {
	_startupCall = call;
}

std::string QuteCom::getStartupCall() const {
	return _startupCall;
}
