/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#include "CimClass.h"

using namespace CimClassNamespace;

// utility function
void VerifyClientNotNull(const WsmanClient *cl)
{
	if(!cl)
	{
		throw CIMClassException("A NULL WsmanClient pointer was provided.", WSMAN_MISSING_INPUT);
	}
}

string CIMCreateErrMsg(const string& name,
					   const string& method,
					   const char* what = NULL)
{
	string ret = "Failed to perform " + name + " " + method + ".";
	if(what)
	{
		ret.append("\n").append(what);
	}
	return ret;
}

CimClass::CimClass(const string& objName,
				   const string& uri,
				   const string& ns,
				   const string& pref):
Serializable(objName, ns, pref),
DeSerializable(objName, ns),
isCreated(false),
resourceUri(uri){}

CimClass::~CimClass(){}

string CimClass::GetObjectName() const
{
	return serializableName;
}

string CimClass::GetResourceUri() const
{
	return resourceUri;
}

EndpointReference CimClass::ToEndpointReference() const
{
	EndpointReference d;
	d.SetResourceUri(GetResourceUri());
	if(isCreated)
	{
		d.SetSelectors(creationData.GetSelectors());
		d.SetAddress(creationData.GetAddress());
	}
	else
	{
		d.SetSelectors(GetSelectorsList());
	}
	return d;
}

void CimClass::Get (const NameValuePairs *selectors, const WsmanClient *cl)
{
	VerifyClientNotNull(cl);
	Deserialize(cl->Get(GetResourceUri(), selectors));
}

void CimClass::Put(const WsmanClient *cl)
{
	VerifyClientNotNull(cl);
	cl->Put(GetResourceUri(), Serialize(), isCreated ? &creationData.GetSelectors() : &GetSelectorsList());
}

void CimClass::Create(const WsmanClient *cl)
{
	VerifyClientNotNull(cl);
	creationData.Deserialize(cl->Create(GetResourceUri(), Serialize()));
	isCreated = true;
}

void CimClass::Delete(const WsmanClient *cl)
{
	VerifyClientNotNull(cl);
	cl->Delete(GetResourceUri(), isCreated ? &creationData.GetSelectors() : &GetSelectorsList());
}

void CimClass::Invoke(const string& functionName,
					  const Serializable* input, DeSerializable* output,
					  const WsmanClient *cl) const
{
	if(!input || !output)
	{
		throw CIMClassException(CIMCreateErrMsg(serializableName, "Invoke"+functionName + "\nFunction input output is missing.").c_str(), WSMAN_MISSING_INPUT);
	}
	VerifyClientNotNull(cl);
	output->Deserialize(cl->Invoke(GetResourceUri(), functionName, input->Serialize(), &GetSelectorsList()));
}

void CimClass::RegisterDefaultWsmanProvider(const WsmanClient *cl)
{
	defaultProvider = cl;
}

const WsmanClient* CimClass::defaultProvider;
