/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015 - 2017 by Kay F. Jahnke                    */
/*                                                                      */
/*    The git repository for this software is at                        */
/*                                                                      */
/*    https://bitbucket.org/kfj/vspline                                 */
/*                                                                      */
/*    Please direct questions, bug reports, and contributions to        */
/*                                                                      */
/*    kfjahnke+vspline@gmail.com                                        */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/*! \file domain.h

    \brief code to perform combined scaling and translation on coordinates
    
    A common requirement is to map coordinates in one range to another
    range, effectively performing a combined scaling and translation.
    Given incoming coordinates in a range [ in_low , in_high ] and a
    desired range for outgoing coordinates of [ out_low , out_high ],
    and an incoming coordinate c, a vspline::domain performs this
    operation:
    
    c' = ( c - in_low ) * scale + out_low
    
    where
    
    scale = ( out_high - out_low ) / ( in_high - in_low )
    
    The code can handle arbitrary dimensions, float and double coordinate
    elementary types, and, optionally, it can perform vectorized operations
    on vectorized coordinates.

    vspline::domain is derived from vspline::unary_functor and can be
    used like any other vspline::unary_functor. A common use case would
    be to access a vspline::evaluator with a different coordinate range
    than the spline's 'natural' coordinates (assuming a 1D spline of floats):
    
    auto _ev = vspline::make_safe_evaluator ( bspl ) ;
    auto ev = vspline::domain ( bspl , 0 , 100 ) + _ev ;
    
    ev.eval ( coordinate , result ) ;
    
    Here, the domain is built over the spline with an incoming range
    of [ 0 , 100 ], so evaluating at 100 will be equivalent to evaluating
    _ev at bspl.upper_limit().
*/

#ifndef VSPLINE_DOMAIN_H
#define VSPLINE_DOMAIN_H

#include <vspline/unary_functor.h>
#include <assert.h>

namespace vspline
{
/// class domain is a coordinate transformation functor. It provides
/// a handy way to translate an arbitrary range of incoming coordinates
/// to an arbitrary range of outgoing coordinates. This is done with a
/// linear translation function. if the source range is [s0,s1] and the
/// target range is [t0,t1], the translation function s->t is:
///
/// t =  ( s - s0 ) * ( t1 - t0 ) / ( s1 - s0 ) + t0 
///
/// In words: the target coordinate's distance from the target range's
/// lower bound is proportional to the source coordinate's distance
/// from the source range's lower bound. Note that this is *not* a
/// gate function: the domain will accept any incoming value and
/// perform the shift/scale operation on it; incoming values outside
/// [ in_low , in_high ] will produce outgoing values outside
/// [ out_low , out_high ].
///
/// The first constructor takes s0, s1, t0 and t1. With this functor,
/// arbitrary mappings of the form given above can be achieved.
/// The second constructor takes a vspline::bspline object to obtain
/// t0 and t1. These are taken as the spline's 'true' range, depending
/// on it's boundary conditions: for periodic splines, this is [0...M],
/// for REFLECT Bcs it's [-0.5,M-0,5], and for 'normal' splines it's
/// [0,M-1]. s0 and s1, the start and end of the domain's coordinate range,
/// can be passed in and default to 0 and 1, which constitutes 'normalized'
/// spline coordinates, where 0 is mapped to the lower end of the 'true'
/// range and 1 to the upper.
///
/// class domain is especially useful for situations where several b-splines
/// cover the same data in different resolution, like in image pyramids.
/// If these different splines are all evaluated with a domain chained to the
/// evaluator which uses a common domain range, they can all be accessed with
/// identical coordinates, even if the spline shapes don't match isotropically.
///
/// The evaluation routine in class domain_type makes sure that incoming
/// values in [ in_low , in_high ] will never produce outgoing values
/// outside [ out_low , out_high ]. If this guarantee is not needed, the
/// 'raw' evaluation routine _eval can be used instead. with _eval, output
/// may overshoot out_high slightly.
///
/// I should mention libeinspline here, which has this facility as a fixed
/// feature in it's spline types. I decided to keep it separate and create
/// class domain instead for those cases where the functionality is needed.

template < typename coordinate_type ,
           int _vsize = vspline::vector_traits<coordinate_type>::size
         >
struct domain_type
: public vspline::unary_functor < coordinate_type ,
                                  coordinate_type ,
                                  _vsize >
{
  typedef vspline::unary_functor < coordinate_type ,
                                   coordinate_type ,
                                   _vsize >
    base_type ;
    
  using base_type::dim_in ;
  using base_type::vsize ;  
  using typename base_type::in_type ;
  using typename base_type::out_type ;
  
  typedef typename base_type::in_ele_type rc_type ;
  typedef typename vigra::TinyVector < rc_type , dim_in > limit_type ;

  // internally, we work with definite TinyVectors:
  
  const limit_type out_low , in_low , out_high , in_high ;
  const limit_type scale ;
  
  /// constructor taking the lower and upper fix points
  /// for incoming and outgoing values
  
  domain_type ( const coordinate_type & _in_low ,
                const coordinate_type & _in_high ,
                const coordinate_type & _out_low ,
                const coordinate_type & _out_high )
  : in_low ( _in_low ) ,
    out_low ( _out_low ) ,
    in_high ( _in_high ) ,
    out_high ( _out_high ) ,
    scale ( ( _out_high - _out_low ) / ( _in_high - _in_low ) )
  { 
    assert ( in_low != in_high && out_low != out_high ) ;
  }

  /// constructor taking the fix points for outgoing values
  /// from a bspline object, and the incoming lower and upper
  /// fix points explicitly

  template < class bspl_type >
  domain_type ( const bspl_type & bspl ,
                const coordinate_type & _in_low = coordinate_type ( 0 ) ,
                const coordinate_type & _in_high = coordinate_type ( 1 ) )
  : in_low ( _in_low ) ,
    in_high ( _in_high ) ,
    out_low ( bspl.lower_limit() ) ,
    out_high ( bspl.upper_limit() ) ,
    scale (   ( bspl.upper_limit() - bspl.lower_limit() )
            / ( _in_high - _in_low ) )
  {
    static_assert ( dim_in == bspl_type::dimension ,
                    "can only create domain from spline if dimensions match" ) ;
    assert ( in_low != in_high ) ;
  }
  
  /// _eval only performs the domain functor's artithmetics. for many use
  /// cases this will be sufficient, but the 'official' eval routine
  /// (below) adds code to make sure that input inside [ in_low , in_high ]
  /// will produce output inside [ out_low , out_high ]. _eval may
  /// produce a value > out_high for _in == in_high due to quantization
  /// errors.

  template < class crd_type >
  void _eval ( const crd_type & _in ,
                     crd_type & _out ) const
  {  
    auto in = wrap ( _in ) ;
    typedef decltype ( in ) nd_crd_t ;
    typedef typename nd_crd_t::value_type component_type ;
    component_type * p_out = (component_type*) ( &_out ) ;
    
    for ( int d = 0 ; d < dim_in ; d++ )
      p_out[d] = ( in[d] - in_low[d] ) * scale[d] + out_low[d] ;
  }

private:

  /// polish sets 'out' to 'subst' where 'patch' indicates.
  /// the unvectorized case will be caught by this overload:
  
  void polish ( rc_type & out ,
                const rc_type & subst ,
                bool patch ) const
  {
    if ( patch )
      out = subst ;
  }
  
#ifdef USE_VC

  /// whereas this template will match vectorized operation
  
  template < class c_type , class mask_type >
  void polish ( c_type & out ,
                const rc_type & subst ,
                mask_type patch ) const
  {
    out ( patch ) = subst ;
  }
  
#endif

public:
  
  /// eval repeats the code of _eval, adding the invocation
  /// of 'polish' which makes sure that, when in == in_high,

  template < class crd_type >
  void eval ( const crd_type & _in ,
                    crd_type & _out ) const
  {  
    auto in = wrap ( _in ) ;
    typedef decltype ( in ) nd_crd_t ;
    typedef typename nd_crd_t::value_type component_type ;
    component_type * p_out = (component_type*) ( &_out ) ;
    
    for ( int d = 0 ; d < dim_in ; d++ )
    {
      p_out[d] = ( in[d] - in_low[d] ) * scale[d] + out_low[d] ;
      polish ( p_out[d] , out_high[d] , in[d] == in_high[d] ) ;
    }
  }

} ;

/// factory function to create a domain_type type object from the
/// desired lower and upper fix point for incoming coordinates and
/// the lower and upper fix point for outgoing coordinates.
/// the resulting functor maps incoming coordinates in the range of
/// [in_low,in_high] to coordinates in the range of [out_low,out_high]

template < class coordinate_type ,
           int _vsize  = vspline::vector_traits<coordinate_type>::size >
vspline::domain_type < coordinate_type , _vsize >
domain ( const coordinate_type & in_low ,
         const coordinate_type & in_high ,
         const coordinate_type & out_low ,
         const coordinate_type & out_high )
{
  return vspline::domain_type < coordinate_type , _vsize >
          ( in_low , in_high , out_low , out_high ) ;
}

/// factory function to create a domain_type type object
/// from the desired lower and upper reference point for incoming
/// coordinates and a vspline::bspline object providing the lower
/// and upper reference for outgoing coordinates
/// the resulting functor maps incoming coordinates in the range of
/// [ in_low , in_high ] to coordinates in the range of
/// [ bspl.lower_limit() , bspl.upper_limit() ]

template < class coordinate_type ,
           class spline_type ,
           int _vsize  = vspline::vector_traits<coordinate_type>::size >
vspline::domain_type < coordinate_type , _vsize >
domain ( const spline_type & bspl ,
         const coordinate_type & in_low = coordinate_type ( 0 ) ,
         const coordinate_type & in_high = coordinate_type ( 1 ) )
{
  return vspline::domain_type < coordinate_type , _vsize >
          ( bspl , in_low , in_high ) ;
}

} ; // namespace vspline

#endif // #ifndef VSPLINE_DOMAIN_H
