/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: control.c,v 1.7 2000/03/19 23:48:47 rich Exp $
 */

#include "xracer.h"
#include "xracer-track.h"
#include "xracer-math.h"
#include "xracer-log.h"
#include "common.h"

/* XXX */
#define SKY_AND_BACKDROP_HIDDEN(seg) 0

/* On-Load function. */
static int
load ()
{
  GLfloat fog_colour[4] = { 1, 1, 1, 1 };

  if (sky_load () < 0) return -1;
  if (backdrop_load () < 0) return -1;
  if (flare_load () < 0) return -1;

  /* Load the scenery objects. */
  if (obj_load () < 0) return -1;

  /* Enable these for fog. */
  glFogi (GL_FOG_MODE, GL_EXP);
  glFogfv (GL_FOG_COLOR, fog_colour);
  glFogf (GL_FOG_DENSITY, 0.01);

  return 0;
}

/* Unload function. */
static int
unload ()
{
  /* Unload the scenery objects. */
  obj_unload ();

  flare_unload ();
  sky_unload ();
  backdrop_unload ();

  return 0;
}

/* Start game function. */
static int
start_game ()
{
  return 0;
}

/* End game function. */
static int
end_game ()
{
  return 0;
}

static void
get_pilot_start_position (GLfloat *posn_rtn)
{
  posn_rtn[0] = -0.01;
  posn_rtn[1] = -16.77;
  posn_rtn[2] = -0.52;
}

static void
get_pilot_start_rotation (GLfloat *rot_rtn)
{
  rot_rtn[0] = 90;
  rot_rtn[1] = 0;
  rot_rtn[2] = 72.91;
}

static int
get_pilot_start_segment ()
{
  return 0;
}

static const struct xrTrackFace *
get_face (int face_num)
{
  return &track_faces[face_num];
}

static int
get_nr_faces ()
{
  return nr_track_faces;
}

static const struct xrTrackEnterPlane *
get_enterplane (int enterplane_num)
{
  return &enterplanes[enterplane_num];
}

static int
get_nr_enterplanes ()
{
  return nr_enterplanes;
}

static int *
map_point_to_faces (const GLfloat *point, int seg)
{
  const struct xrTrackBSPNode *node = track_root;
#define MAX_NR_FACES 32
  static int faces[MAX_NR_FACES+1];
  int i, j, k, facenum;

  /* Recurse down the BSP until we find the terminal node
   * corresponding to this point.
   */
  while (node->type == nodetype_interior)
    {
      if (node->u.i.split_axis == 'x')
	{
	  if (point[1] <= node->u.i.split)
	    node = node->u.i.subtree0;
	  else
	    node = node->u.i.subtree1;
	}
      else /* node->u.i.split_axis == 'y' */
	{
	  if (point[0] <= node->u.i.split)
	    node = node->u.i.subtree0;
	  else
	    node = node->u.i.subtree1;
	}
    }

  /* Examine each face in turn to determine whether the point lies
   * inside the face. If so, push the index onto our private list.
   */
  i = 0;
  for (j = 0; j < node->u.t.nr_faces; ++j)
    {
      facenum = node->u.t.faces[j];
      for (k = 0; k < 6; ++k)
	{
	  if (! xrPointInsidePlane (track_faces[facenum].plane_coefficient[k],
				    point))
	    break;
	}

      if (k == 6)
	{
	  xrLogAssert (i < MAX_NR_FACES);

	  faces[i++] = facenum;
	}
    }

  /* Terminate the list with a -1. */
  faces[i++] = -1;

  return faces;
}

static inline void
display_scenery (int seg)
{
  /* Display scenery visible from the current segment. */
  switch (seg)
    {
    case 0:
      obj_StartGantry_display ();
      obj_StartGantryLegs_display ();
      break;
    case 1:
      obj_StartGantry_display ();
      obj_StartGantryLegs_display ();
      break;
    case 4:
      obj_SnowMountain_display ();
      obj_SnowMountain01_display ();
      obj_SnowMountain02_display ();
      break;
    case 5:
      obj_SnowMountain_display ();
      obj_SnowMountain01_display ();
      obj_SnowMountain02_display ();
      break;
    case 6:
      obj_TunnelEntrance_display ();
      obj_SnowMountain_display ();
      obj_SnowMountain01_display ();
      obj_SnowMountain02_display ();
      obj_SnowMountain03_display ();
      break;
    case 7:
      obj_TunnelEntrance_display ();
      obj_SnowMountain_display ();
      obj_SnowMountain01_display ();
      obj_SnowMountain02_display ();
      obj_SnowMountain03_display ();
      break;
    case 8:
      obj_TunnelEntrance_display ();
      obj_SnowMountain03_display ();
      break;
    case 24:
      obj_StartGantry_display ();
      obj_StartGantryLegs_display ();
      break;
    case 25:
      obj_StartGantry_display ();
      obj_StartGantryLegs_display ();
      break;
    }
  /* XXX */
  obj_TrackScenery_display ();
}

/* This is where the action happens. This is the display function for
 * the track.
 */
static void
display (const struct xrPlayer *player)
{
  static GLfloat ambient[4] = { 0.2, 0.2, 0.2, 0 };
  static GLfloat origin[4] = { 0, 0, 0, 0 };

  const GLfloat *posn = xrPlayerGetPosition (player);
  int seg = xrPlayerGetSegment (player);

  glShadeModel (GL_SMOOTH);
  /*glShadeModel (GL_FLAT);*/

  glEnable (GL_TEXTURE_2D);
  glDepthMask (GL_FALSE);

  if (!SKY_AND_BACKDROP_HIDDEN(seg))
    {
      /* Draw the sky. */
      sky_display (posn);

      /* Draw the backdrop. */
      glEnable (GL_BLEND);
      glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      backdrop_display (posn);
      glDisable (GL_BLEND);
    }

  glEnable (GL_DEPTH_TEST);
  glDepthMask(GL_TRUE);
  glEnable (GL_LIGHTING);
  glEnable (GL_LIGHT0);
  glLightfv (GL_LIGHT0, GL_POSITION, origin);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);

  /* Enable this for fog. */
  glEnable (GL_FOG);

  /* Draw the track and scenery in here. */
  display_scenery (seg);

  /* XXX Draw the opponents in here. Need to make a callback into main prog. */


  /* Disable everything we just enabled. */
  glDisable (GL_FOG);
  glDisable (GL_TEXTURE_2D);
  glDisable (GL_LIGHT0);
  glDisable (GL_DEPTH_TEST);
  glShadeModel (GL_FLAT);
  glDisable (GL_LIGHTING);

  if (!SKY_AND_BACKDROP_HIDDEN(seg))
    {
      /* Draw lens flares */
      flare_display (player);
    }
}

/* Control structure. This symbol is exported. */
struct xrTrackControl track =
{
  name: "simple",

  load: load,
  unload: unload,
  start_game: start_game,
  end_game: end_game,

  get_pilot_start_position: get_pilot_start_position,
  get_pilot_start_rotation: get_pilot_start_rotation,
  get_pilot_start_segment: get_pilot_start_segment,

  get_enterplane: get_enterplane,
  get_nr_enterplanes: get_nr_enterplanes,

  get_face: get_face,
  get_nr_faces: get_nr_faces,

  map_point_to_faces: map_point_to_faces,

  display: display
  /*display_precipitation: display_precipitation*/
};
